%--------------------------------------------------------------------------------------
% Cross-validation of the predictive model built from the German corpus.
%--------------------------------------------------------------------------------------

function cross_validation_german()

clear all;
close all; 

modelPalateX = [-5.5 -4.1 -3.9 -3.763 -3.515 -3.323 -3.105 -2.934 -2.75  -2.574 -2.402 -2.281 -2.065 -1.981 -1.758 -1.487 -1.33 -1.108 -0.769 -0.41 0];
modelPalateY = [-1.5 -1.5 -1.5 -1.500 -1.406 -1.296 -0.937 -0.693 -0.406 -0.168  0.007  0.121  0.238  0.278  0.293  0.278  0.271 0.234  0.187  0.125 0]; 

NUM_EXAMPLES = 25;
NUM_TONGUE_POINTS = 20;

% 20 x-coordinates for the tongue contour of each of the 25 examples.

allContourX = [
-3.9402	-3.4302	-2.8298	-2.2019	-1.5713	-0.9417	-0.3129	0.3173	0.9409	1.5461	2.0999	2.5305	2.8918	3.1589	3.4905	3.7969	3.9071	3.5153	3.0345	2.8811
-2.9224	-2.3604	-1.7881	-1.2277	-0.7003	-0.2232	0.2938	0.8347	1.4057	1.9390	2.4486	2.8460	3.2717	3.5359	3.8243	4.0300	4.0462	3.7382	3.2836	2.8594
-3.4622	-2.9485	-2.3722	-1.7799	-1.1980	-0.6556	-0.1449	0.4092	1.0015	1.5946	2.1233	2.5971	2.9560	3.2555	3.5947	3.8217	3.7863	3.4695	3.0268	2.6140
-3.9988	-3.5161	-2.9609	-2.3982	-1.8428	-1.2890	-0.7303	-0.1696	0.3877	0.9411	1.4432	1.8905	2.2593	2.5151	2.6742	2.8068	2.9673	2.9144	2.5589	2.2750
-3.9988	-3.6092	-3.0916	-2.5345	-1.9740	-1.4182	-0.8665	-0.3142	0.2390	0.7997	1.3525	1.8629	2.2757	2.5326	2.6800	2.7483	2.8721	2.8012	2.5077	2.1856
-3.7248	-3.2693	-2.7195	-2.1823	-1.6572	-1.1292	-0.5795	-0.0225	0.5352	1.0727	1.5401	1.9157	2.2012	2.5182	2.8549	3.2641	3.5483	3.4662	3.0801	2.8692
-1.8534	-1.4516	-1.0174	-0.5011	0.0172	0.5455	1.1095	1.6832	2.2397	2.7049	3.0548	3.2229	3.2569	3.2034	3.1069	3.1368	3.0748	2.8552	2.4369	2.1896
-3.4715	-2.9816	-2.3935	-1.8063	-1.2737	-0.7994	-0.3130	0.2674	0.8869	1.5106	2.0784	2.5434	2.8696	2.9914	3.0435	3.0477	3.0580	2.8711	2.4878	2.0768
-3.7912	-3.2827	-2.7097	-2.1363	-1.6303	-1.1440	-0.6314	-0.0685	0.5242	1.1011	1.6406	2.1254	2.5573	2.7978	2.8927	2.9873	3.1976	3.1303	2.7988	2.4896
-3.6680	-3.1694	-2.5782	-1.9783	-1.3856	-0.8037	-0.2337	0.3454	0.9360	1.5359	2.1165	2.5924	2.9402	3.1957	3.3644	3.4807	3.6367	3.3935	2.9645	2.7773
-3.7644	-3.3775	-3.0527	-2.7547	-2.4244	-1.9770	-1.4351	-0.8637	-0.3084	0.1807	0.6201	0.9677	1.3047	1.5974	1.7031	1.9080	2.2408	2.4027	2.3494	2.2806
-2.0554	-1.8914	-1.5154	-1.0499	-0.5296	0.0127	0.5649	1.1234	1.6458	2.0964	2.4399	2.6079	2.6896	2.6520	2.4716	2.2422	2.1479	2.1005	1.8624	1.5627
-3.4715	-3.1190	-2.7426	-2.3811	-1.9574	-1.4166	-0.8421	-0.2615	0.2810	0.6951	1.0363	1.2389	1.4023	1.5828	1.8711	2.3647	2.7396	2.8435	2.5820	2.5845
-3.3543	-3.1237	-2.8560	-2.5405	-2.1218	-1.6013	-1.0559	-0.5391	-0.0481	0.4122	0.8859	1.3204	1.6748	1.9308	2.0984	2.2044	2.3176	2.3781	2.1960	1.7918
-3.8498	-3.3466	-2.7596	-2.1610	-1.6166	-1.0824	-0.5676	0.0014	0.5885	1.1828	1.7515	2.2682	2.6821	3.0306	3.4303	3.7290	3.8099	3.5263	3.0238	2.6158
-3.7058	-3.3043	-2.7788	-2.2660	-1.8055	-1.2849	-0.7147	-0.1237	0.4479	1.0094	1.5103	2.0238	2.4062	2.6800	2.8870	3.2637	3.5089	3.3957	2.9883	2.6223
-3.2836	-2.8024	-2.3038	-1.7541	-1.2045	-0.6609	-0.1522	0.3808	0.9412	1.5081	2.0486	2.5095	2.8661	3.1508	3.3366	3.4320	3.5470	3.5391	3.1498	2.8602
-3.6472	-3.3181	-2.9816	-2.6571	-2.2664	-1.7684	-1.2151	-0.6504	-0.1209	0.3533	0.7382	1.0414	1.2980	1.4909	1.6108	1.9286	2.3326	2.6411	2.3980	2.2634
-3.7381	-3.3426	-2.9362	-2.5755	-2.1337	-1.5791	-0.9890	-0.4386	0.0288	0.4622	0.7797	1.0386	1.1641	1.1869	1.1974	1.3355	1.7165	1.8569	1.5634	1.5792
-3.7639	-3.3308	-2.8146	-2.2841	-1.7455	-1.2018	-0.6364	-0.0789	0.4564	0.9682	1.4246	1.7809	2.0351	2.1950	2.2920	2.6088	2.8238	2.7926	2.4539	2.3299
-3.5295	-3.0862	-2.5118	-1.9139	-1.3213	-0.7170	-0.1472	0.4536	1.0660	1.6757	2.2173	2.6710	2.9108	3.0079	2.9557	2.8673	2.8413	2.7079	2.3922	2.1736
-2.5123	-2.1223	-1.5579	-0.9685	-0.4324	0.0385	0.5099	1.0611	1.6538	2.2554	2.7776	3.2167	3.4904	3.6598	3.7391	3.9371	3.8724	3.5053	3.0197	2.9312
-2.9224	-2.6870	-2.3195	-1.8173	-1.2896	-0.7481	-0.2083	0.3316	0.8569	1.3414	1.7145	1.9782	2.0888	2.0838	1.8870	1.7709	1.9028	2.0638	2.0167	2.0142
-3.7428	-3.3957	-2.8085	-2.1918	-1.5919	-0.9876	-0.3728	0.2492	0.8710	1.4879	1.9988	2.4646	2.9054	3.1342	3.2602	3.5042	3.5743	3.3745	2.8513	2.4582
-2.4537	-2.1255	-1.6695	-1.1872	-0.6838	-0.1786	0.3371	0.8536	1.3687	1.8566	2.2970	2.6799	2.9575	3.1751	3.3313	3.4859	3.6057	3.6190	3.4265	3.1134
];

% 20 y-coordinates for the tongue contour of each of the 25 examples.

allContourY = [
-1.8047	-1.4372	-1.2659	-1.2173	-1.2048	-1.1744	-1.1616	-1.1524	-1.2254	-1.4014	-1.7022	-2.1584	-2.6738	-3.2444	-3.7693	-4.3176	-4.9017	-5.3892	-5.7969	-6.3986
-2.5787	-2.4224	-2.3087	-2.1480	-1.9043	-1.5690	-1.3014	-1.0827	-1.0207	-1.2468	-1.5307	-1.9556	-2.3501	-2.8683	-3.3744	-3.9171	-4.4962	-4.9800	-5.3383	-5.7362
-2.6555	-2.3538	-2.1940	-2.1096	-1.9673	-1.7139	-1.4023	-1.1887	-1.1367	-1.2031	-1.4693	-1.8341	-2.3084	-2.8252	-3.3181	-3.8679	-4.4505	-4.9492	-5.3514	-5.7818
-1.8773	-1.6170	-1.5289	-1.5038	-1.4142	-1.3110	-1.2819	-1.3329	-1.4158	-1.5161	-1.7717	-2.1133	-2.5350	-3.0362	-3.5754	-4.1147	-4.6461	-5.1907	-5.6275	-6.1129
-2.2630	-1.8644	-1.6560	-1.5897	-1.6238	-1.5420	-1.4330	-1.3276	-1.2271	-1.2122	-1.3153	-1.5435	-1.9182	-2.4161	-2.9587	-3.5145	-4.0593	-4.6092	-5.0882	-5.5490
-2.5400	-2.2099	-2.0908	-1.9355	-1.7297	-1.5331	-1.4303	-1.5044	-1.5876	-1.7590	-2.0716	-2.4823	-2.9688	-3.4356	-3.8833	-4.2718	-4.7459	-5.2930	-5.6962	-6.2041
-2.2835	-1.8680	-1.4907	-1.2311	-0.9759	-0.7423	-0.6253	-0.6742	-0.8100	-1.1474	-1.6020	-2.1521	-2.7259	-3.3011	-3.8673	-4.4445	-5.0156	-5.5439	-5.9417	-6.4544
-2.7543	-2.3610	-2.1405	-1.9229	-1.5897	-1.1749	-0.7742	-0.5354	-0.4417	-0.4974	-0.7619	-1.1684	-1.6976	-2.3144	-2.9408	-3.5670	-4.1899	-4.7866	-5.2777	-5.7350
-2.5576	-2.2562	-2.1013	-1.9438	-1.6298	-1.2849	-0.9839	-0.7995	-0.7906	-0.9105	-1.1606	-1.5034	-1.9075	-2.4443	-3.0325	-3.6212	-4.1713	-4.7578	-5.2481	-5.7549
-2.7184	-2.3926	-2.2834	-2.3092	-2.2038	-2.0509	-1.8556	-1.6895	-1.5706	-1.5304	-1.6881	-2.0434	-2.5249	-3.0690	-3.6475	-4.2371	-4.8101	-5.3411	-5.7620	-6.3204
-2.4151	-1.9894	-1.5135	-1.0206	-0.5493	-0.1868	-0.0080	0.0224	-0.1288	-0.4257	-0.7985	-1.2561	-1.7227	-2.2181	-2.7840	-3.3186	-3.7879	-4.3342	-4.8988	-5.4678
-2.1221	-1.5927	-1.1805	-0.8706	-0.6626	-0.5254	-0.4427	-0.4794	-0.6689	-0.9960	-1.4304	-1.9641	-2.5165	-3.0744	-3.6052	-4.1077	-4.6589	-5.2168	-5.7116	-6.1700
-2.7937	-2.3171	-1.8592	-1.3903	-0.9762	-0.7415	-0.6365	-0.7496	-0.9690	-1.3899	-1.8740	-2.4273	-2.9972	-3.5614	-4.0497	-4.3755	-4.8198	-5.3796	-5.9016	-6.4822
-2.1052	-1.6038	-1.1208	-0.6687	-0.3178	-0.1397	-0.0909	-0.2759	-0.5271	-0.8297	-1.1117	-1.4519	-1.8711	-2.3594	-2.8813	-3.4219	-3.9605	-4.5090	-5.0081	-5.3841
-2.6787	-2.3542	-2.2973	-2.3096	-2.0597	-1.7876	-1.4870	-1.2938	-1.1693	-1.2536	-1.4420	-1.7420	-2.1775	-2.6673	-3.1155	-3.6321	-4.2172	-4.6904	-5.0203	-5.4573
-2.4002	-1.9697	-1.7077	-1.4211	-1.0570	-0.7917	-0.6340	-0.6348	-0.7732	-0.9400	-1.2551	-1.5496	-1.9955	-2.5199	-3.0719	-3.5219	-4.0488	-4.6283	-5.0520	-5.5108
-3.1959	-2.8870	-2.6101	-2.4525	-2.2949	-2.1209	-1.8613	-1.6680	-1.5614	-1.5726	-1.7496	-2.0793	-2.5227	-3.0180	-3.5587	-4.1205	-4.6797	-5.2319	-5.6459	-6.1277
-2.3494	-1.8852	-1.4264	-0.9589	-0.5499	-0.2891	-0.1707	-0.1894	-0.3954	-0.7007	-1.1198	-1.6006	-2.1043	-2.6397	-3.1916	-3.6556	-4.0558	-4.5100	-4.9954	-5.5309
-1.9427	-1.5055	-1.0771	-0.6060	-0.2182	-0.0107	0.0218	-0.1973	-0.5619	-0.9671	-1.4615	-1.9927	-2.5718	-3.1644	-3.7566	-4.3304	-4.7804	-5.3277	-5.8352	-6.4153
-2.4308	-2.0678	-1.8313	-1.6273	-1.4455	-1.2803	-1.2589	-1.3545	-1.5452	-1.7896	-2.1256	-2.5633	-3.0717	-3.6137	-4.1715	-4.6399	-5.1598	-5.7181	-6.1693	-6.7082
-2.8214	-2.4161	-2.2124	-2.0742	-1.9155	-1.8113	-1.5918	-1.4707	-1.4407	-1.5043	-1.7743	-2.1860	-2.7485	-3.3492	-3.9597	-4.5664	-5.1787	-5.7727	-6.2844	-6.8350
-3.4179	-2.9530	-2.7176	-2.5448	-2.2514	-1.8580	-1.4671	-1.2016	-1.0595	-1.1722	-1.4848	-1.9110	-2.4566	-3.0473	-3.6511	-4.2269	-4.8041	-5.2910	-5.6676	-6.2458
-1.4854	-0.9983	-0.6080	-0.4093	-0.3030	-0.2844	-0.2423	-0.2736	-0.3968	-0.6310	-1.0237	-1.4940	-2.0207	-2.5617	-3.0593	-3.5737	-4.0941	-4.6070	-5.1327	-5.6500
-2.4196	-1.9230	-1.7434	-1.6625	-1.4981	-1.3513	-1.2708	-1.2542	-1.2321	-1.2488	-1.6043	-2.0141	-2.4529	-3.0191	-3.6265	-4.1952	-4.8013	-5.3799	-5.7167	-6.1884
-2.8428	-2.4433	-2.2035	-2.0164	-1.8942	-1.7829	-1.7343	-1.7005	-1.7378	-1.8944	-2.1657	-2.5146	-2.9504	-3.4184	-3.9123	-4.4067	-4.9097	-5.4239	-5.9011	-6.3101
];

% Four EPG indices for each of the 25 examples.

allEpgData = [
3.5000	-0.2716	2.4328	1.3462
0.0000	0.0000	0.0000	0.0000
0.2500	-0.3468	-0.3204	0.3266
4.3750	-0.4410	1.2144	2.2420
3.5000	-0.5879	0.4531	2.9630
1.2500	0.0000	-0.8372	1.6332
0.2500	0.0000	-0.3468	0.3266
0.8750	-0.1734	-1.0857	0.9519
1.3750	-0.1470	-1.3999	1.5095
1.0000	0.0000	-1.1839	1.1152
4.6250	0.0982	-1.7264	2.3541
1.2500	0.0000	-1.5307	0.9799
3.1250	-0.2946	-1.8965	2.4334
5.0000	0.1389	-2.2689	1.6332
0.8750	-0.1470	-1.0105	1.0476
2.5000	-0.1964	-1.7982	2.2304
0.0000	0.0000	0.0000	0.0000
4.5000	-0.1327	-1.8246	2.3261
5.2500	0.0435	-1.8998	1.8642
2.5000	-0.5879	-0.8284	2.5010
0.0000	0.0000	0.0000	0.0000
0.0000	0.0000	0.0000	0.0000
4.2500	0.0000	-2.1186	1.6332
0.0000	0.0000	0.0000	0.0000
0.0000	0.0000	0.0000	0.0000
];

% Five OPG distances each of the 25 examples in cm.
% Note: The first (most anterior) distance d1 was weighted by 1/(1.0 cm + d1)

allOpgData = [
0.0000	0.9070	1.5970	1.4310	1.1850
0.5398	2.2220	2.3990	2.3660	1.5840
0.4992	2.0170	2.2500	2.1940	1.5090
0.0000	1.1284	1.6178	1.5063	1.3351
0.1823	1.3870	1.7710	1.6810	1.3210
0.3958	1.4530	1.7740	1.6720	1.4520
0.7241	1.9030	2.0020	1.9860	1.1910
0.4869	1.6730	1.8090	1.7640	0.6700
0.4055	1.3940	1.5450	1.3670	0.7390
0.4334	2.0780	2.3960	2.2260	1.7930
0.2852	0.2430	0.3300	0.2050	0.2490
0.6882	1.2980	1.3470	1.1320	0.5350
0.4734	0.8310	1.0520	0.8870	0.7720
0.3132	0.1950	0.3350	0.2620	0.4810
0.4111	1.8630	2.0790	2.0270	1.3920
0.3351	0.9370	1.1350	0.9530	0.6320
0.5950	2.4360	2.6360	2.5020	2.0130
0.3056	0.2660	0.4860	0.3590	0.4170
0.2045	0.1830	0.3710	0.2680	0.5250
0.2748	1.1840	1.6470	1.4740	1.3840
0.4627	1.9340	2.3320	2.0830	1.7240
0.6593	2.7520	2.8620	2.8490	2.2350
0.5937	0.2380	0.5610	0.4820	0.2520
0.2565	1.3610	1.7990	1.6320	1.3290
0.6263	2.1700	2.4610	2.2970	1.8570
];

% Concatenation of EPG and OPG data

allEpgOpgData = [ allEpgData allOpgData ];

% Add a column of ones to the left of the data matrices.

allEpgData = [ ones(NUM_EXAMPLES, 1) allEpgData ];
allOpgData = [ ones(NUM_EXAMPLES, 1) allOpgData ];
allEpgOpgData = [ ones(NUM_EXAMPLES, 1) allEpgOpgData ];


% *****************************************************************************
% Create training and testing examples.
% *****************************************************************************

testingExample = 1;     % 1 .. 25

% Create the training data (24 examples).

trainingContourX = allContourX;
trainingContourY = allContourY;
trainingEpgData = allEpgData;
trainingOpgData = allOpgData;
trainingEpgOpgData = allEpgOpgData;

% Remove the rows with the testing example.

trainingContourX(testingExample, :) = [];   
trainingContourY(testingExample, :) = [];
trainingEpgData(testingExample, :) = [];
trainingOpgData(testingExample, :) = []; 
trainingEpgOpgData(testingExample, :) = [];

% Create the testing data (one example).

testingContourX = allContourX(testingExample, :);
testingContourY = allContourY(testingExample, :);
testingEpgData = allEpgData(testingExample, :);
testingOpgData = allOpgData(testingExample, :);
testingEpgOpgData = allEpgOpgData(testingExample, :);

% *****************************************************************************
% Create models from training examples in terms of the parameters of the
% multiple linear regression models.
% The size of the paramyX and paramsY matrices is (NUM_TONGUE_POINTS x N),
% where N = 4+1 for EPG, N = 5+1 for OPG, and N = 4+5+1 for both.
% *****************************************************************************

paramsEpgX = []; 
paramsEpgY = []; 

for i = 1:NUM_TONGUE_POINTS
    paramsEpgX = [paramsEpgX; lsqlin(trainingEpgData, trainingContourX(:,i))']; 
    paramsEpgY = [paramsEpgY; lsqlin(trainingEpgData, trainingContourY(:,i))'];
end 

% *******************************************************************

paramsOpgX = []; 
paramsOpgY = []; 

for i = 1:NUM_TONGUE_POINTS
    paramsOpgX = [paramsOpgX; lsqlin(trainingOpgData, trainingContourX(:,i))']; 
    paramsOpgY = [paramsOpgY; lsqlin(trainingOpgData, trainingContourY(:,i))'];
end 

% *******************************************************************

paramsEpgOpgX = []; 
paramsEpgOpgY = []; 

for i = 1:NUM_TONGUE_POINTS
    paramsEpgOpgX = [paramsEpgOpgX; lsqlin(trainingEpgOpgData, trainingContourX(:,i))']; 
    paramsEpgOpgY = [paramsEpgOpgY; lsqlin(trainingEpgOpgData, trainingContourY(:,i))'];
end 

% *******************************************************************
% Calculate and display the predicted tongue contours for the testing
% example based on EPG, OPG, and both.
% *******************************************************************

disp(' ');

% *********************************************************
% Prediction with EPG.
% *********************************************************

predictedContourX = (paramsEpgX * testingEpgData')';     % Make it a row vector.
predictedContourY = (paramsEpgY * testingEpgData')';     % Make it a row vector.

subplot(1,3,1);
axis equal;
hold on; 
plot(modelPalateX, modelPalateY, '-black');
plot(testingContourX, testingContourY, ':black');
plot(predictedContourX, predictedContourY, '-red');
hold off;

error_mm = 10.0*getError_cm(testingContourX, testingContourY, predictedContourX, predictedContourY);
disp(['Error EPG (mm) = ' num2str(error_mm)]);

% *********************************************************
% Prediction with OPG.
% *********************************************************

predictedContourX = (paramsOpgX * testingOpgData')';     % Make it a row vector.
predictedContourY = (paramsOpgY * testingOpgData')';     % Make it a row vector.

subplot(1,3,2);
axis equal;
hold on; 
plot(modelPalateX, modelPalateY, '-black');
plot(testingContourX, testingContourY, ':black');
plot(predictedContourX, predictedContourY, '-red');
hold off;

error_mm = 10.0*getError_cm(testingContourX, testingContourY, predictedContourX, predictedContourY);
disp(['Error OPG (mm) = ' num2str(error_mm)]);

% *********************************************************
% Prediction with combined EPG+OPG.
% *********************************************************

predictedContourX = (paramsEpgOpgX * testingEpgOpgData')';     % Make it a row vector.
predictedContourY = (paramsEpgOpgY * testingEpgOpgData')';     % Make it a row vector.

subplot(1,3,3);
axis equal;
hold on; 
plot(modelPalateX, modelPalateY, '-black');
plot(testingContourX, testingContourY, ':black');
plot(predictedContourX, predictedContourY, '-red');
hold off;

error_mm = 10.0*getError_cm(testingContourX, testingContourY, predictedContourX, predictedContourY);
disp(['Error EPG+OPG (mm) = ' num2str(error_mm)]);


% Just temporary...

disp('Predicted contour based on OPG data:');
predictedContourX = (paramsOpgX * testingOpgData')
predictedContourY = (paramsOpgY * testingOpgData')


% *****************************************************************************
% *****************************************************************************

function error_cm = getError_cm(testingContourX, testingContourY, predictedContourX, predictedContourY)

NUM_POINTS = 20;
allDistances = zeros(1, NUM_POINTS);
allProjections = zeros(1, NUM_POINTS - 1);
minDistance = zeros(1, NUM_POINTS);

% Pick a point on the example countour and run through all predicted points. 
% Find eculidean distances with each point as well as 
% shortest distance of the point with all line segments. 
% Find the minimum of the two and thats the curve fit error for the point. 

for i = 1:NUM_POINTS
    currentPointX = testingContourX(i);
    currentPointY = testingContourY(i);

    for k = 1:NUM_POINTS
        dx = currentPointX - predictedContourX(k); 
        dy = currentPointY - predictedContourY(k); 

        allDistances(k) = sqrt((dx * dx) + (dy * dy)); 
    end

    % calculate the projection of currentPoint on the example countour 
    % on all line segments of predicted contour.
    
    for k = 1:(NUM_POINTS-1)
        x0 = currentPointX; 
        y0 = currentPointY; 
        x1 = predictedContourX(k); 
        y1 = predictedContourY(k);
        x2 = predictedContourX(k + 1); 
        y2 = predictedContourY(k + 1);

        u = ((x0 - x1) * (x2 - x1) + (y0 - y1) * (y2 - y1)) / ((x2 - x1) * (x2 - x1) + (y2 - y1) * (y2 - y1));

        % The point is left of the line segment and doesn't make a
        % projection.
        if (u < 0.0)  
            dx = x0 - x1; 
            dy = y0 - y1; 
            euclidDistance = sqrt((dx * dx) + (dy *dy)); 
        
        %  the point is right of the line segment and doesn't make a projection
        elseif (u > 1.0) 
            dx = x0 - x2; 
            dy = y0 - y2; 
            euclidDistance = sqrt((dx * dx) + (dy *dy));
        
        % make a projection on the line segment and then find the euclidean
        % distance.
        else    
            projX = x1 + u * ( x2 - x1); 
            projY = y1 + u * ( y2 - y1);   

            dx = x0 - projX; 
            dy = y0 - projY; 
            euclidDistance = sqrt((dx * dx) + (dy * dy));
        end
        
        allProjections(k) = euclidDistance; 
    end
    
    % Find the minimum distance.
    minDistance(i) = min([allDistances allProjections]);
end
    
error_cm = mean(minDistance);    

% *****************************************************************************

