%--------------------------------------------------------------------------------------
% Cross-validation of the predictive model built from the English corpus.
%--------------------------------------------------------------------------------------

function cross_validation_english()

clear all;
close all; 

modelPalateX = [-5.5 -4.1 -3.9 -3.763 -3.515 -3.323 -3.105 -2.934 -2.75  -2.574 -2.402 -2.281 -2.065 -1.981 -1.758 -1.487 -1.33 -1.108 -0.769 -0.41 0];
modelPalateY = [-1.5 -1.5 -1.5 -1.500 -1.406 -1.296 -0.937 -0.693 -0.406 -0.168  0.007  0.121  0.238  0.278  0.293  0.278  0.271 0.234  0.187  0.125 0]; 

NUM_EXAMPLES = 25;
NUM_TONGUE_POINTS = 20;

% 20 x-coordinates for the tongue contour of each of the 25 examples.

allContourX = [
    % extracted from English corpus
-3.2777	-3.0116	-2.6270	-2.1209	-1.5786	-1.0135	-0.4229	0.0951	0.5883	1.0405	1.4722	1.7842	1.8966	1.9703	1.9725	1.9725	1.9725	1.9430	1.7653	1.8631
-2.8123	-2.5530	-2.1203	-1.6269	-1.0926	-0.5456	-0.0047	0.5177	0.9542	1.3487	1.7431	2.1274	2.3337	2.3602	2.3867	2.4133	2.4282	2.2629	2.0793	2.0004
-2.3184	-2.0235	-1.6188	-1.0576	-0.4843	0.0826	0.6150	1.1602	1.6167	2.0684	2.4546	2.7000	2.8592	3.0295	3.2357	3.3304	3.2265	2.9395	2.5725	2.3849
-1.6739	-1.3718	-0.8858	-0.3269	0.2543	0.8399	1.4223	1.9915	2.5175	2.9569	3.2575	3.4725	3.5763	3.5763	3.5763	3.5043	3.1745	2.8176	2.5787	2.4825
-1.8809	-1.6034	-1.1582	-0.6589	-0.1350	0.4022	0.9152	1.4194	1.8993	2.3619	2.6925	3.0703	3.3410	3.4786	3.4786	3.4786	3.4524	3.2681	2.9905	2.7130
-2.2091	-1.8932	-1.3498	-0.7756	-0.1854	0.4263	1.0281	1.6093	2.1299	2.6064	2.9934	3.1686	3.2297	3.1617	3.0411	2.9962	2.6099	2.1758	2.0567	2.0567
-3.3028	-3.0794	-2.6634	-2.1566	-1.6126	-1.0070	-0.3858	0.1634	0.6695	1.1596	1.6273	1.9273	1.9473	1.9044	1.7447	1.6462	1.5324	1.5098	1.5282	1.7286
-2.6466	-2.3378	-1.8684	-1.3766	-0.7908	-0.2027	0.3744	0.9056	1.3666	1.8024	2.1642	2.2755	2.2755	2.1008	1.9045	1.7615	1.6575	1.5302	1.4112	1.5098
-2.3184	-2.0965	-1.6174	-1.0719	-0.4835	0.1022	0.6612	1.2026	1.7326	2.2070	2.6619	2.9958	3.0411	3.0411	2.9766	2.8484	2.5388	2.1470	1.8491	1.9473
-1.8809	-1.5910	-1.0761	-0.5070	0.0585	0.6160	1.1901	1.7600	2.3026	2.7941	3.1468	3.4430	3.4786	3.5303	3.6423	3.6974	3.4893	3.0858	2.7853	2.6036
-1.7715	-1.4601	-1.0216	-0.4443	0.1704	0.7871	1.3988	1.9952	2.5618	3.0091	3.3667	3.5576	3.5880	3.5561	3.4256	3.2819	3.0347	2.6213	2.3774	2.2755
-2.7560	-2.4606	-1.9866	-1.3817	-0.7577	-0.1723	0.4309	0.9987	1.5850	2.1875	2.7308	3.1341	3.2599	3.2599	3.2949	3.3091	3.0234	2.6166	2.2755	2.2755
-2.9865	-2.7272	-2.2820	-1.7354	-1.1609	-0.5681	0.0276	0.5924	1.1019	1.5661	1.9873	2.2396	2.3656	2.3756	2.4825	2.4825	2.3900	2.3158	2.1037	2.0450
-2.3184	-2.0803	-1.6844	-1.1095	-0.4813	0.1463	0.7428	1.3097	1.8799	2.4636	2.9787	3.2623	3.3693	3.2643	3.0894	2.8156	2.3956	2.0393	1.9473	2.0567
-4.3966	-3.7794	-3.1429	-2.4949	-1.8638	-1.2327	-0.5869	0.0681	0.7206	1.3652	1.9819	2.5553	2.9483	3.0411	2.9764	2.7716	2.4093	1.9849	1.8380	1.9473
-3.7521	-3.1588	-2.5450	-1.9218	-1.2911	-0.6603	-0.0382	0.5753	1.1910	1.8021	2.3794	2.8639	3.1271	3.1388	3.1388	3.0312	2.7172	2.2219	1.9413	2.0450
-3.4122	-3.0612	-2.6293	-2.1262	-1.5850	-1.0275	-0.4831	0.0351	0.4558	0.8654	1.3017	1.6869	1.9820	2.1652	2.1661	2.1019	1.9605	1.7477	1.6192	1.6192
-3.0841	-2.7490	-2.3140	-1.7315	-1.1642	-0.5996	-0.0191	0.5628	1.1407	1.6997	2.1591	2.5757	2.7439	2.8224	2.8224	2.6822	2.4191	2.0447	1.7028	1.6192
-3.8222	-3.2892	-2.7122	-2.1247	-1.5466	-0.9719	-0.3930	0.1343	0.6600	1.1905	1.6982	2.0870	2.3361	2.4124	2.4124	2.2703	2.0927	1.8824	1.6596	1.6468
-2.7560	-2.3721	-1.8123	-1.2263	-0.6453	-0.0679	0.4904	0.9836	1.4612	1.9660	2.3951	2.6601	2.7130	2.6967	2.5869	2.5031	2.2003	1.8715	1.7286	1.8380
-3.5909	-3.1041	-2.4908	-1.8615	-1.2484	-0.6278	-0.0004	0.5506	1.0356	1.5375	2.0288	2.3841	2.6509	2.7402	2.6471	2.5569	2.3686	2.0863	1.8379	1.8780
-2.6466	-2.3960	-1.9386	-1.3897	-0.7882	-0.1543	0.4596	1.0483	1.6525	2.2061	2.6021	2.8321	2.9318	2.7976	2.5971	2.3966	2.1264	2.0173	1.9671	2.0567
-3.3028	-2.8652	-2.2854	-1.6820	-1.0446	-0.3879	0.2409	0.8513	1.3927	1.7999	2.1783	2.4304	2.6155	2.6565	2.6976	2.7130	2.5687	2.3045	1.9974	2.0567
-3.2628	-3.0454	-2.5756	-2.0155	-1.4352	-0.8580	-0.2786	0.3007	0.8473	1.3338	1.7476	2.1351	2.4293	2.4939	2.4739	2.3120	2.0815	1.7953	1.6593	1.7686
-3.7128	-3.2594	-2.6758	-2.0848	-1.4945	-0.9047	-0.3148	0.2516	0.8043	1.3475	1.8670	2.3004	2.6547	2.7406	2.7406	2.6606	2.4340	2.0868	1.8655	1.9749


];

% 20 y-coordinates for the tongue contour of each of the 25 examples.

allContourY = [
    % extracted from English corpus
-2.2819	-1.7497	-1.3030	-0.9925	-0.7490	-0.5673	-0.5318	-0.7963	-1.1164	-1.5008	-1.9065	-2.4039	-2.9871	-3.5774	-4.1723	-4.7673	-5.3622	-5.9503	-6.5077	-7.0945
-2.2365	-1.7585	-1.4106	-1.1516	-0.9957	-0.9617	-1.0942	-1.2846	-1.6278	-2.0223	-2.4168	-2.8186	-3.3289	-3.8861	-4.4433	-5.0006	-5.5567	-6.0868	-6.6063	-7.1586
-2.6580	-2.1664	-1.7721	-1.6736	-1.6736	-1.7381	-1.9371	-2.1057	-2.4380	-2.7791	-3.2001	-3.7127	-4.2634	-4.8106	-5.3442	-5.8832	-6.4436	-6.9363	-7.3767	-7.9082
-3.1738	-2.6704	-2.3509	-2.1845	-2.1015	-2.0801	-2.1474	-2.2810	-2.5237	-2.9072	-3.4114	-3.9563	-4.5265	-5.1137	-5.7008	-6.2709	-6.7476	-7.2105	-7.7461	-8.3147
-2.6580	-2.1954	-1.9138	-1.7098	-1.5869	-1.5642	-1.7103	-1.9018	-2.1479	-2.4254	-2.8493	-3.2340	-3.6952	-4.2103	-4.7498	-5.2892	-5.8255	-6.3267	-6.7893	-7.2519
-2.3298	-1.8034	-1.5267	-1.3109	-1.1422	-1.1267	-1.2250	-1.4076	-1.7330	-2.1138	-2.5864	-3.1672	-3.7781	-4.3745	-4.9689	-5.5692	-6.0424	-6.4765	-7.0755	-7.6894
-2.0017	-1.4216	-0.9479	-0.5677	-0.2537	-0.0769	-0.1131	-0.4113	-0.7910	-1.1921	-1.6114	-2.1602	-2.7905	-3.4198	-4.0316	-4.6566	-5.2797	-5.9110	-6.5415	-7.1425
-1.5642	-1.0496	-0.6798	-0.3443	-0.2158	-0.1842	-0.3491	-0.6231	-1.0073	-1.4192	-1.8886	-2.4748	-3.0750	-3.6415	-4.2084	-4.7902	-5.3812	-5.9672	-6.5505	-7.1425
-2.2205	-1.6712	-1.3230	-1.1076	-1.0340	-1.0926	-1.2786	-1.5200	-1.7859	-2.1417	-2.5170	-3.0049	-3.5872	-4.1802	-4.7696	-5.3404	-5.8434	-6.2866	-6.7809	-7.3613
-2.6580	-2.1730	-1.9372	-1.8923	-1.9670	-2.1064	-2.1111	-2.1781	-2.3434	-2.6392	-3.0906	-3.5830	-4.1478	-4.7182	-5.2814	-5.8493	-6.3609	-6.7697	-7.2536	-7.7988
-2.8767	-2.3466	-1.9080	-1.6972	-1.6216	-1.5713	-1.6733	-1.8435	-2.0902	-2.5166	-3.0183	-3.5997	-4.2169	-4.8319	-5.4357	-6.0339	-6.6020	-7.0612	-7.6246	-8.2363
-2.3298	-1.7774	-1.3983	-1.2456	-1.2478	-1.4718	-1.6421	-1.9066	-2.1261	-2.2859	-2.5969	-3.0627	-3.6602	-4.2875	-4.9113	-5.5238	-6.0665	-6.5419	-7.0621	-7.6894
-2.5185	-1.9906	-1.6194	-1.3837	-1.2346	-1.2060	-1.2060	-1.3626	-1.6711	-2.0397	-2.4609	-2.9930	-3.5747	-4.1693	-4.7518	-5.3474	-5.9339	-6.5186	-7.0750	-7.6593
-2.1111	-1.5254	-1.0395	-0.8446	-0.8368	-0.9225	-1.1260	-1.4094	-1.6841	-1.9084	-2.2674	-2.8280	-3.4412	-4.0622	-4.6702	-5.2386	-5.7099	-6.2277	-6.8503	-7.4706
-2.2205	-1.9890	-1.8292	-1.7089	-1.5220	-1.3321	-1.2360	-1.2957	-1.3889	-1.5160	-1.7455	-2.0628	-2.5816	-3.2189	-3.8691	-4.4926	-5.0398	-5.5360	-6.1686	-6.8144
-1.5332	-1.3301	-1.1861	-1.0956	-1.0956	-1.0956	-1.1822	-1.3281	-1.4649	-1.6200	-1.8483	-2.2521	-2.8167	-3.4452	-4.0759	-4.6959	-5.2361	-5.6220	-6.1612	-6.7833
-1.5642	-1.1037	-0.7226	-0.4373	-0.2324	-0.2080	-0.4037	-0.6627	-1.0570	-1.4667	-1.8456	-2.2672	-2.7621	-3.3116	-3.8908	-4.4648	-5.0248	-5.5635	-6.1257	-6.7050
-2.2205	-1.7372	-1.3804	-1.3454	-1.4749	-1.6384	-1.7285	-1.8123	-1.9219	-2.0661	-2.4336	-2.8488	-3.4070	-3.9825	-4.5708	-5.1259	-5.6521	-6.1012	-6.5796	-7.1425
-1.5198	-1.2772	-1.1917	-1.1917	-1.2684	-1.3898	-1.4898	-1.7280	-1.9876	-2.2365	-2.5299	-2.9471	-3.4783	-4.0534	-4.6409	-5.1949	-5.7519	-6.2988	-6.8408	-7.4263
-1.2360	-0.7979	-0.6892	-0.7434	-0.8586	-0.9884	-1.1763	-1.5050	-1.8559	-2.1660	-2.5690	-3.0992	-3.6794	-4.2706	-4.8529	-5.4397	-5.9427	-6.4359	-7.0008	-7.5800
-1.6595	-1.2785	-1.1332	-1.1126	-0.9645	-0.9314	-1.0011	-1.3008	-1.7050	-2.0878	-2.4804	-2.9999	-3.5694	-4.1944	-4.8143	-5.4391	-6.0378	-6.6025	-7.1797	-7.7847
-1.8923	-1.3136	-0.8769	-0.5619	-0.3614	-0.3610	-0.4659	-0.7014	-0.8889	-1.1898	-1.6706	-2.2595	-2.8813	-3.4979	-4.0994	-4.7009	-5.2628	-5.8872	-6.5148	-7.1425
-1.5642	-1.0901	-0.7820	-0.5221	-0.3688	-0.3610	-0.5097	-0.7539	-1.1208	-1.6355	-2.1679	-2.7719	-3.3956	-4.0517	-4.7079	-5.3648	-5.9961	-6.5934	-7.1610	-7.7988
-2.3158	-1.7724	-1.4253	-1.2848	-1.2091	-1.1129	-1.0301	-1.0592	-1.2276	-1.5528	-1.9666	-2.4033	-2.9023	-3.4840	-4.0585	-4.6209	-5.1588	-5.6665	-6.2259	-6.8003
-1.6292	-1.2802	-1.1917	-1.1917	-1.2141	-1.2510	-1.2878	-1.4341	-1.6432	-1.8760	-2.1551	-2.5448	-3.0123	-3.5893	-4.1803	-4.7634	-5.3049	-5.7822	-6.3061	-6.8794
    
    
];

% Four EPG indices for each of the 25 examples.

allEpgData = [
    % extracted from English corpus
2.5000	-0.1964	-2.0636	2.0391
1.2500	0.0000	-1.4779	1.2505
0.0000	0.0000	0.0000	0.0000
0.0000	0.0000	0.0000	0.0000
0.0000	0.0000	0.0000	0.0000
0.2500	0.0000	-0.3468	0.3266
3.7500	-0.2654	-2.8318	1.4419
2.3750	-0.0345	-2.5574	1.2785
0.5000	0.0000	-0.6407	0.6533
0.0000	0.0000	0.0000	0.0000
0.0000	0.0000	0.0000	0.0000
0.8750	-0.1470	-1.0105	1.0476
0.7500	0.0264	-0.9123	0.8843
1.3750	0.0982	-1.6513	1.1829
1.2500	-0.2940	-1.2265	1.4419
3.2500	-0.4754	0.5920	2.5967
4.5000	0.0000	-1.9222	1.9598
0.7500	0.0000	-0.8372	0.9799
4.6250	-0.1470	1.1081	2.3773
2.6250	-0.0982	-1.8477	2.2420
4.8750	-0.0982	1.0329	2.3773
2.5000	-0.1815	-2.5282	1.3462
3.7500	-0.1327	-1.9974	2.2701
1.5000	0.0000	-1.5469	1.5772
4.6250	-0.4410	1.2293	2.1068
];

% Five OPG distances each of the 25 examples in cm.
% Note: The first (most anterior) distance d1 was weighted by 1/(1.0 cm + d1)

allOpgData = [
    % extracted from English corpus
0.4289	0.5500	0.7550	0.6200	0.6470
0.5238	0.9490	1.1370	1.0510	1.0780
0.6676	1.5680	1.7720	1.7450	1.6920
0.7482	2.5350	2.6310	2.5430	2.1970
0.7121	1.9920	2.1170	1.9850	1.6610
0.7295	1.5260	1.6610	1.5060	1.1650
0.3880	0.3460	0.4070	0.1900	0.2540
0.5645	0.6620	0.6750	0.3710	0.2360
0.6715	1.3470	1.4400	1.2540	1.0940
0.7174	1.9750	2.0880	2.0000	1.8760
0.7438	2.1650	2.2600	2.0260	1.6960
0.5953	1.1010	1.3700	1.3530	1.5090
0.5779	1.2520	1.4740	1.3750	1.2670
0.6540	1.1680	1.1340	0.9630	0.9170
0.3377	1.2580	1.4970	1.3790	1.2690
0.0476	0.7270	1.1690	1.2030	1.1580
0.2212	0.2540	0.3780	0.2210	0.4880
0.5192	0.9830	1.4570	1.6400	1.7430
0.0000	0.7300	1.3330	1.4720	1.5340
0.6816	0.3100	0.6280	0.8330	0.9170
0.0000	0.5320	1.0380	0.9390	0.9540
0.6005	0.7700	0.7760	0.5110	0.3190
0.2571	0.3900	0.6190	0.4620	0.3380
0.4647	0.8450	1.2540	1.2140	1.0160
0.0000	0.6970	1.2460	1.3290	1.3100    
];

% Concatenation of EPG and OPG data

allEpgOpgData = [ allEpgData allOpgData ];

% Add a column of ones to the left of the data matrices.

allEpgData = [ ones(NUM_EXAMPLES, 1) allEpgData ];
allOpgData = [ ones(NUM_EXAMPLES, 1) allOpgData ];
allEpgOpgData = [ ones(NUM_EXAMPLES, 1) allEpgOpgData ];


% *****************************************************************************
% Create training and testing examples.
% *****************************************************************************

testingExample = 8;     % 1 .. 25

% Create the training data (24 examples).

trainingContourX = allContourX;
trainingContourY = allContourY;
trainingEpgData = allEpgData;
trainingOpgData = allOpgData;
trainingEpgOpgData = allEpgOpgData;

% Remove the rows with the testing example.

trainingContourX(testingExample, :) = [];   
trainingContourY(testingExample, :) = [];
trainingEpgData(testingExample, :) = [];
trainingOpgData(testingExample, :) = []; 
trainingEpgOpgData(testingExample, :) = [];

% Create the testing data (one example).

testingContourX = allContourX(testingExample, :);
testingContourY = allContourY(testingExample, :);
testingEpgData = allEpgData(testingExample, :);
testingOpgData = allOpgData(testingExample, :);
testingEpgOpgData = allEpgOpgData(testingExample, :);

% *****************************************************************************
% Create models from training examples in terms of the parameters of the
% multiple linear regression models.
% The size of the paramyX and paramsY matrices is (NUM_TONGUE_POINTS x N),
% where N = 4+1 for EPG, N = 5+1 for OPG, and N = 4+5+1 for both.
% *****************************************************************************

paramsEpgX = []; 
paramsEpgY = []; 

for i = 1:NUM_TONGUE_POINTS
    paramsEpgX = [paramsEpgX; lsqlin(trainingEpgData, trainingContourX(:,i))']; 
    paramsEpgY = [paramsEpgY; lsqlin(trainingEpgData, trainingContourY(:,i))'];
end 

% *******************************************************************

paramsOpgX = []; 
paramsOpgY = []; 

for i = 1:NUM_TONGUE_POINTS
    paramsOpgX = [paramsOpgX; lsqlin(trainingOpgData, trainingContourX(:,i))']; 
    paramsOpgY = [paramsOpgY; lsqlin(trainingOpgData, trainingContourY(:,i))'];
end 

% *******************************************************************

paramsEpgOpgX = []; 
paramsEpgOpgY = []; 

for i = 1:NUM_TONGUE_POINTS
    paramsEpgOpgX = [paramsEpgOpgX; lsqlin(trainingEpgOpgData, trainingContourX(:,i))']; 
    paramsEpgOpgY = [paramsEpgOpgY; lsqlin(trainingEpgOpgData, trainingContourY(:,i))'];
end 

% *******************************************************************
% Calculate and display the predicted tongue contours for the testing
% example based on EPG, OPG, and both.
% *******************************************************************

disp(' ');

% *********************************************************
% Prediction with EPG.
% *********************************************************

predictedContourX = (paramsEpgX * testingEpgData')';     % Make it a row vector.
predictedContourY = (paramsEpgY * testingEpgData')';     % Make it a row vector.

subplot(1,3,1);
axis equal;
hold on; 
plot(modelPalateX, modelPalateY, '-black');
plot(testingContourX, testingContourY, ':black');
plot(predictedContourX, predictedContourY, '-red');
hold off;

error_mm = 10.0*getError_cm(testingContourX, testingContourY, predictedContourX, predictedContourY);
disp(['Error EPG (mm) = ' num2str(error_mm)]);

% *********************************************************
% Prediction with OPG.
% *********************************************************

predictedContourX = (paramsOpgX * testingOpgData')';     % Make it a row vector.
predictedContourY = (paramsOpgY * testingOpgData')';     % Make it a row vector.

subplot(1,3,2);
axis equal;
hold on; 
plot(modelPalateX, modelPalateY, '-black');
plot(testingContourX, testingContourY, ':black');
plot(predictedContourX, predictedContourY, '-red');
hold off;

error_mm = 10.0*getError_cm(testingContourX, testingContourY, predictedContourX, predictedContourY);
disp(['Error OPG (mm) = ' num2str(error_mm)]);

% *********************************************************
% Prediction with combined EPG+OPG.
% *********************************************************

predictedContourX = (paramsEpgOpgX * testingEpgOpgData')';     % Make it a row vector.
predictedContourY = (paramsEpgOpgY * testingEpgOpgData')';     % Make it a row vector.

subplot(1,3,3);
axis equal;
hold on; 
plot(modelPalateX, modelPalateY, '-black');
plot(testingContourX, testingContourY, ':black');
plot(predictedContourX, predictedContourY, '-red');
hold off;

error_mm = 10.0*getError_cm(testingContourX, testingContourY, predictedContourX, predictedContourY);
disp(['Error EPG+OPG (mm) = ' num2str(error_mm)]);


% Just temporary...

disp('Predicted contour based on OPG data:');
predictedContourX = (paramsOpgX * testingOpgData'); 
predictedContourY = (paramsOpgY * testingOpgData'); 


% *****************************************************************************
% *****************************************************************************

function error_cm = getError_cm(testingContourX, testingContourY, predictedContourX, predictedContourY)

NUM_POINTS = 20;
allDistances = zeros(1, NUM_POINTS);
allProjections = zeros(1, NUM_POINTS - 1);
minDistance = zeros(1, NUM_POINTS);

% Pick a point on the example countour and run through all predicted points. 
% Find eculidean distances with each point as well as 
% shortest distance of the point with all line segments. 
% Find the minimum of the two and thats the curve fit error for the point. 

for i = 1:NUM_POINTS
    currentPointX = testingContourX(i);
    currentPointY = testingContourY(i);

    for k = 1:NUM_POINTS
        dx = currentPointX - predictedContourX(k); 
        dy = currentPointY - predictedContourY(k); 

        allDistances(k) = sqrt((dx * dx) + (dy * dy)); 
    end

    % calculate the projection of currentPoint on the example countour 
    % on all line segments of predicted contour.
    
    for k = 1:(NUM_POINTS-1)
        x0 = currentPointX; 
        y0 = currentPointY; 
        x1 = predictedContourX(k); 
        y1 = predictedContourY(k);
        x2 = predictedContourX(k + 1); 
        y2 = predictedContourY(k + 1);

        u = ((x0 - x1) * (x2 - x1) + (y0 - y1) * (y2 - y1)) / ((x2 - x1) * (x2 - x1) + (y2 - y1) * (y2 - y1));

        % The point is left of the line segment and doesn't make a
        % projection.
        if (u < 0.0)  
            dx = x0 - x1; 
            dy = y0 - y1; 
            euclidDistance = sqrt((dx * dx) + (dy *dy)); 
        
        %  the point is right of the line segment and doesn't make a projection
        elseif (u > 1.0) 
            dx = x0 - x2; 
            dy = y0 - y2; 
            euclidDistance = sqrt((dx * dx) + (dy *dy));
        
        % make a projection on the line segment and then find the euclidean
        % distance.
        else    
            projX = x1 + u * ( x2 - x1); 
            projY = y1 + u * ( y2 - y1);   

            dx = x0 - projX; 
            dy = y0 - projY; 
            euclidDistance = sqrt((dx * dx) + (dy * dy));
        end
        
        allProjections(k) = euclidDistance; 
    end
    
    % Find the minimum distance.
    minDistance(i) = min([allDistances allProjections]);
end
    
error_cm = mean(minDistance);   


% *****************************************************************************

