function [data, max_amp] = preprocess_data(data, spectral_smoothing, ...
    remove_data_below_bckgd_noise, normalize, min_percent_area)

    nPol = length(data.polarAngles);
    nAzi = length(data.azimuthalAngles);

    % smooth data
    if spectral_smoothing
        data.amplitude = smoothdata(data.amplitude, 3, 'gaussian',12);
        data.bckgdNoise = smoothdata(data.bckgdNoise, 2, 'gaussian',125);
    end
    
    % remove data below background noise
    if remove_data_below_bckgd_noise
        for a = 1:nAzi
            amp = squeeze(data.amplitude(:,a,:));
            if nPol == 1
                amp = amp';
            end
            amp(amp < data.bckgdNoise) = nan;
            data.amplitude(:,a,:) = amp;
        end
        
        % remove the low frequencies below 200 Hz
        % is checked if there is enough frequencies: if there is less than
        % 100 then a band averaging have been done and this is not
        % necessary
        if size(data.amplitude, 3) > 100
            idx_200Hz = round(200 / data.freqStep);
            data.amplitude(:,:,1:idx_200Hz) = nan;
        end
    end
    
    % normalize
    max_amp = max(max(data.amplitude, [], 1), [], 2);
    if normalize
        data.amplitude = data.amplitude - repmat(max_amp, nPol, nAzi, 1);
    end
    
    % remove the data with not enough spatial coverage
    dAz2 = abs(data.azimuthalAngles(2) - data.azimuthalAngles(1)) * pi / 180 / 2;
    if nPol == 1
        dPo2 = 15 * pi / 180 / 2;
    else
        dPo2 = abs(data.polarAngles(2) - data.polarAngles(1)) * pi / 180 / 2;
    end
    ref_area = zeros(data.nbFreqs, 1);
    for p = 1:nPol
        po = (90 - data.polarAngles(p)) * pi / 180;
        for a = 1:nAzi
            % polar boundaries
            po1 = max(0, po - dPo2);
            po2 = min(pi, po + dPo2);
            
            % azimutal boundaries
            az = data.azimuthalAngles(a) * pi / 180;
            az1 = max(-pi, az - dAz2);
            az2 = min(pi, az + dAz2);
            
            % surface correcponding to the current point
            surf_pt = (cos(po1) - cos(po2)) * (az2 - az1);

            % take into account only the area elements with non NAN data
            vec_area = surf_pt * ~isnan(squeeze(data.amplitude(p,a,:)));
            ref_area = ref_area + vec_area;
        end
    end
    under_min_area = ref_area < min_percent_area * max(ref_area) / 100;
    data.amplitude(:,:,under_min_area) = nan;
end
