clear all

%% parameters

% Three types of representation of the directivity can be plotted: 
%
%   - Directivity maps
%   - Directivity index
%   - Directivity ballons
%
% To select one of them uncomment the line defining the corresponding value
% of "data_to_plot" below and comment the other ones

%**************************************************************************
% DIRECTIVITY MAPS
% data_to_plot = 'MAPS';
%
vertical_map =                      false;
% if set to false the directivity maps of the horizontal plane are
% displayed

%**************************************************************************
% DIRECTIVITY INDEX
% data_to_plot = 'INDEX'; 
%
DI_ref_in_front =                   false;

%**************************************************************************
% DIRECTIVTY BALLOON
%
data_to_plot = 'BALLOON';
%
% when the ballon are display, one browses the frequency by pressing
% "ENTER" or any other key. The starting frequency can be set here:
start_freq_balloon = 100;
% to select a specific point of view to display the directivity ballons,
% uncomment the relevant value
%
% view_type = 'TOP_LEFT_SIDE';
% view_type = 'TOP_LEFT_SIDE_MORE_CENTERED';
view_type = 'TOP_RIGHT_SIDE_MORE_CENTERED';
% view_type = 'LEFT_SIDE';
% view_type = 'TOP';
%
% in order to print as image the ballons of the 3d octave band averaging
print_oct_band =                    false;

%**************************************************************************
% TO EXCLUDE SOME PART OF THE DATA
%
mid_hor_plane_only =                false;
exclude_back =                      true;

%**************************************************************************
% DATA PROCESSING
% 
correct_anomalies =                 true;
spectral_smoothing =                true;
remove_data_below_bckgd_noise =     true;
normalize =                         true;
interpolateData =                   false;
average_oct_bands =                 false;
% select the type of band if a band averaging have been chosen
% band_type = 'octave';
band_type = '3d_octave';

% minimal percentage of the area measured required to consider the  
% data as valid: the data having less than min_perc_area data above the 
% noise threshold are integrally replaced by NAN
% (0 take all, 100 take only if all the surface is covered)
min_perc_area = 0;

caxis_lim = [-40 0];

%% get file path and load data

[file, path] = uigetfile('*.mat', 'Select directivity data', 'directivity_data');

fprintf('%s\n', file)

if path ~= 0
    load([path file])
else
    return
end

% set parameters for simulation data
if strcmp(file(end-7:end-4), 'simu')
    correct_anomalies =                 false;
    spectral_smoothing =                false;
    remove_data_below_bckgd_noise =     false;
end

%% Pre-processing

switch data_to_plot
    % for the index computation the background noise must be kept
    % and the data must not be normalized
    case 'INDEX'
        if ~average_oct_bands
            remove_data_below_bckgd_noise =     false;
            normalize =                         false;
        end
    case 'BALLOON'
        normalize =                         true;
end

% correct invalid data
if correct_anomalies
    data = correct_data(data, file);
end

% keep only the horizontal plane
if mid_hor_plane_only && length(data.polarAngles) > 1
    idx_pol_90 = find(data.polarAngles == 90);
    data.polarAngles = 90;
    data.amplitude = data.amplitude(idx_pol_90, :, :);
    data.bckgdNoise = data.bckgdNoise(idx_pol_90, :);
end

% exclude the data in the back
if exclude_back
    idx_front = find(data.azimuthalAngles >= -90 & data.azimuthalAngles <= 90);
    data.amplitude = data.amplitude(:, idx_front, :);
    data.azimuthalAngles = data.azimuthalAngles(idx_front);
end

% interpolate data
if interpolateData
    data = interpolate_directivity_data(data, 'cubic', 5);
end

% preprocess data
[data, max_amp] = preprocess_data(data, spectral_smoothing, ...
    remove_data_below_bckgd_noise, normalize, min_perc_area);

% average on octave or 3d octave bands
if average_oct_bands
    [data, freq_vec] = average_bands(data, band_type);
else
    % generate frequency vector
    freq_vec = (1:data.nbFreqs)*data.freqStep;
end

nPol = size(data.amplitude, 1);
nAzi = size(data.amplitude, 2);

switch data_to_plot
    
    %% plot directivity maps
    case 'MAPS'

        % check if there are enough data in the vertical plane
        if vertical_map && (length(data.polarAngles) == 1)
            fprintf('Not enough data in the vertical plane\n')
            return
        end
        
        if vertical_map && (nAzi > nPol)
            nAngle = nAzi / 2;
        else
            nAngle = nPol;
        end

        h = figure('position', [680   770   860   210]) ;
        hold off
        for ii = 1:nAngle
            
            % extract data to plot
            if vertical_map
                if nAzi > nPol
                    dataToPlot = [flipud(squeeze(data.amplitude(:,ii,:))) ; ...
                        squeeze(data.amplitude(2:end, ii + floor(nAzi / 2), :))];
                    fprintf('Azi1: %d, azi2: %d\n', data.azimuthalAngles(ii), ...
                        data.azimuthalAngles(ii + floor(nAzi / 2)));
                    polarAngles = [-fliplr(data.polarAngles) data.polarAngles(2:end)];
                else
                    dataToPlot = flipud(squeeze(data.amplitude(:,ii,:)));
                    polarAngles = -fliplr(data.polarAngles);
                end
            else
                dataToPlot = squeeze(data.amplitude(ii,:,:));
            end

            %% plot Directivity map

            if vertical_map
                imagesc(freq_vec/1000, polarAngles, dataToPlot);
            else
                imagesc(freq_vec/1000, data.azimuthalAngles, dataToPlot); 
            end

            axis xy

            if normalize
                caxis(caxis_lim)
            else
                caxis([min(min(dataToPlot)) max(max(dataToPlot))])
            end
            colormap(viridis)
            colorbar
            xlabel('f (kHz)')

            if vertical_map
                ylabel('Polar angle (deg)')
                title(strrep(sprintf('%s azimutal angle %d deg',file, data.azimuthalAngles(ii)), '_', ' '))
            else
                ylabel('Azimutal angle (deg)')
                title(strrep(sprintf('%s polar angle %d deg', file, data.polarAngles(ii)), '_', ' '))
            end

            pause
        end

    %% plot directivity index
    
    case 'INDEX'
        
        % compute directivity index with the background noise (it must be
        % kept before)
        [di_noise, maxPol, maxAz, av_amp_noise] = directivity_index(data, DI_ref_in_front);
        
        % reprocess data removing the background noise if it is a
        % measurement
        if ~(strcmp(file(end-7:end-4), 'simu') || average_oct_bands)
            [data, max_amp] = preprocess_data(data, false, true, false, 0);
        end
        
        % recompute the directivity index without the background noise
        [di, maxPol, maxAz, av_amp] = directivity_index(data, DI_ref_in_front);
        
        % compute the difference of average amplitude computed including
        % and excluding noise
        diff_av_amp = av_amp - av_amp_noise;
        
        % exclude the directivity indexes for which the differencein
        % average amplitude is greater than 1 dB
        di_noise_reliable = nan(size(di));
        di_noise_reliable(diff_av_amp < 1) = di_noise(diff_av_amp < 1);
        
        h = figure;
        
        subplot 511
        
%         plot(freq_vec/1000, di)
        hold on
%         plot(freq_vec/1000, di_noise,'k')
        plot(freq_vec/1000, di_noise_reliable, 'r', 'linewidth', 2)
        grid on
        xlabel('f (kHz)')
        xlim([0 max(freq_vec/1000)])
        ylabel('DI (dB)')
        title(strrep(file, '_', ' '))
        
        subplot 512
        
        plot(freq_vec/1000, av_amp)
        hold on
        plot(freq_vec/1000, av_amp_noise)
        grid on
        xlabel('f (kHz)')
        xlim([0 max(freq_vec/1000)])
        ylabel('Average amp (dB)')
        
        subplot 513
        
        plot(freq_vec/1000, diff_av_amp)
        hold on
        plot([freq_vec(1) freq_vec(end)]/1000, [1 1])
        grid on
        xlabel('f (kHz)')
        xlim([0 max(freq_vec/1000)])
        ylabel('average diff (dB)')
        
        subplot 514
        
        plot(freq_vec/1000, maxPol, '.')
        set(gca, 'YDir','reverse')
        set(gca, 'Ytick', [0 45 90 135 165])
        xlabel('f (kHz)')
        grid on
        xlim([0 max(freq_vec/1000)])
        if nPol == 1
            ylim([0 165]);
        else
            ylim([min(data.polarAngles) max(data.polarAngles)])
        end
        ylabel('Polar angle (deg)')
        
        subplot 515
        
        plot(freq_vec/1000, maxAz, '.')
        set(gca, 'Ytick', [-180 -90 0 90 180])
        xlabel('f (kHz)')
        grid on
        xlim([0 max(freq_vec/1000)])
        ylim([min(data.azimuthalAngles) max(data.azimuthalAngles)])
        ylabel('Azimutal angle (deg)')
        
        % save the data
        if mid_hor_plane_only
            prefix = 'di_hp_';
        else
            prefix = 'di_';
        end
        if average_oct_bands
            prefix = [prefix band_type '_'];
        end
        
        %% Plot directivity ballon
        
    case 'BALLOON'
        
        % Check if there is enough data in the vertical plane
        if length(data.polarAngles) == 1
            fprintf('Not enough data in the vertical plane\n')
            return
        end
            
        max_scale = 50;
        angle_labels = true;
        
        pol = repmat(data.polarAngles', 1, nAzi) * pi / 180;
        azi = repmat(data.azimuthalAngles, nPol, 1) * pi / 180;
        
        % generate mesh coordinates
        x_mesh = max_scale .* sin(pol) .* cos(azi);
        y_mesh = max_scale .* sin(pol) .* sin(azi);
        z_mesh = max_scale .* cos(pol);
        
        if average_oct_bands
            idx_step = 1;
        else
            idx_step = 1;
        end
        
        % if necessary, create the directory to save the images
        if average_oct_bands && print_oct_band
            dir_name = [pwd '/' file(1:end-4)];
            if exist(dir_name, 'dir') == 0
                mkdir(dir_name)
            end
            angle_labels = false;
        end
        
        if average_oct_bands
            idx_start = 1;
        else
            idx_start = round(start_freq_balloon / data.freqStep);
        end
        
        figure
        for f = idx_start:idx_step:data.nbFreqs
            amp = squeeze(data.amplitude(:,:,f)) + max_scale;

            % generate field coordinates
            x = amp .* sin(pol) .* cos(azi);
            y = amp .* sin(pol) .* sin(azi);
            z = amp .* cos(pol);
            
            % plot mesh
            hold off
            mesh(x_mesh, y_mesh, z_mesh, 'edgecolor', [211 211 211]./255,...
                'facecolor', 'none', 'facealpha', 0.5)
            hold on
            mesh(x, y, z, amp, 'linestyle', 'none', 'facecolor', 'flat', ...
                'facealpha', 1)
            caxis(caxis_lim + 50)
            colormap(viridis)
            axis equal
            axis off

            switch view_type
                case 'TOP_LEFT_SIDE'
                    view(70, 15);
                    if angle_labels
                        text(15, 0, -48, '165', 'HorizontalAlignment','center')
                        text(0, 0, 55, '0', 'HorizontalAlignment','center')
                        text(0, -55, 0, '-90', 'HorizontalAlignment','center')
                        text(0, 57, 0, '90', 'HorizontalAlignment','center')
                        text(55, 0, 0, '0', 'HorizontalAlignment','center')
                        text(-50, 0, 0, '180', 'HorizontalAlignment','center')
                    end
                case 'TOP_LEFT_SIDE_MORE_CENTERED'
                    view(80, 20);
                    if angle_labels
                        text(15, 0, -48, '165', 'HorizontalAlignment','center')
                        text(0, 0, 55, '0', 'HorizontalAlignment','center')
                        text(0, -55, 0, '-90', 'HorizontalAlignment','center')
                        text(0, 57, 0, '90', 'HorizontalAlignment','center')
                        text(55, 0, 0, '0', 'HorizontalAlignment','center')
                        text(-50, 0, 0, '180', 'HorizontalAlignment','center')
                    end
                case 'TOP_RIGHT_SIDE_MORE_CENTERED'
                    view(100, 20);
                    if angle_labels
                        text(15, 0, -48, '165', 'HorizontalAlignment','center')
                        text(0, 0, 55, '0', 'HorizontalAlignment','center')
                        text(0, -55, 0, '-90', 'HorizontalAlignment','center')
                        text(0, 57, 0, '90', 'HorizontalAlignment','center')
                        text(55, 0, 0, '0', 'HorizontalAlignment','center')
                        text(-50, 0, 0, '180', 'HorizontalAlignment','center')
                    end
                case 'LEFT_SIDE'
                    view(0, 0);
                    if angle_labels
                        text(15, -55, -48, '165', 'HorizontalAlignment','center')
                        text(0, -55, 55, '0', 'HorizontalAlignment','center')
                        text(0, -55, 0, '-90', 'HorizontalAlignment','center')
                        text(55, -55, 0, '0', 'HorizontalAlignment','center')
                        text(-50, -55, 0, '180', 'HorizontalAlignment','center')
                    end
                case 'TOP'
                    view(0, 90);
                    if angle_labels
                        text(15, 0, -48, '165', 'HorizontalAlignment','center')
                        text(0, 0, 55, '0', 'HorizontalAlignment','center')
                        text(0, -55, 0, '-90', 'HorizontalAlignment','center')
                        text(0, 57, 0, '90', 'HorizontalAlignment','center')
                        text(55, 0, 0, '0', 'HorizontalAlignment','center')
                        text(-50, 0, 0, '180', 'HorizontalAlignment','center')
                    end
            end

            if average_oct_bands && print_oct_band
                print(gcf, '-dpng', [pwd '/' file(1:end-4) '/'...
                    '3doct_' num2str(round(freq_vec(f))) '_Hz.png'])
            else
                title(strrep(sprintf('%s %3.f Hz', file, freq_vec(f)), '_', ' '))
                pause
            end
        end
end

%% Function to compute the directivity index

function [di, maxPol, maxAz, av_amp] = directivity_index(data, ref_in_front)
ref_area = zeros(data.nbFreqs, 1);
di = zeros(data.nbFreqs, 1);

nAz = length(data.azimuthalAngles);
nPo = length(data.polarAngles);

dAz2 = abs(data.azimuthalAngles(2) - data.azimuthalAngles(1)) * pi / 180 / 2;
if nPo == 1
    dPo2 = 15 * pi / 180 / 2;
else
    dPo2 = abs(data.polarAngles(2) - data.polarAngles(1)) * pi / 180 / 2;
end

idxPo90 = find(data.polarAngles == 90);
idxAz0 = find(data.azimuthalAngles == 0);

for p = 1:nPo
    po = data.polarAngles(p) * pi / 180;
    for a = 1:nAz
        % polar boundaries
        po1 = max(0, po - dPo2);
        po2 = min(pi, po + dPo2);
        
        % azimutal boundaries
        az = data.azimuthalAngles(a) * pi / 180;
        az1 = max(-pi, az - dAz2);
        az2 = min(pi, az + dAz2);
        
        % surface correcponding to the current point
        surf_pt = (cos(po1) - cos(po2)) * (az2 - az1);
        
        % extract amplitude
        amp = db2mag(squeeze(data.amplitude(p,a,:)));
        
        % take into account only the area elements with non NAN data
        vec_area = surf_pt * ~isnan(amp);
        ref_area = ref_area + vec_area;
        amp(isnan(amp)) = 0;
        di = di + amp .* vec_area;
    end
end

% compute the average amplitude
av_amp = 20*log10(di ./ ref_area);

if ref_in_front
    di = 20*log10(ref_area .* db2mag(squeeze(data.amplitude(idxPo90, idxAz0, :))) ./ di);
    
    maxPol = 90*ones(1, data.nbFreqs);
    maxAz = zeros(1, data.nbFreqs);
else
    % locate the maximum of amplitude on polar positions
    [maxAmpPol, idxMaxPol] = max(data.amplitude, [], 1);
    maxAmpPol = squeeze(maxAmpPol(1,:,:));
    
    % locate the maximum of amplitude on azimutal positions
    [maxAmpAz, idxMaxAz] = max(maxAmpPol, [], 1);
    
    % compute the directivity index
    maxAmpAz = maxAmpAz';
    di = 20*log10(ref_area .* db2mag(maxAmpAz) ./ di);
    
    % identify the azimutal position of the maximum
    maxAz = data.azimuthalAngles(idxMaxAz);
    
    % identify the polar position of the maximum
%     idxMaxAz = squeeze(idxMaxAz(1, 1, :));
    maxPolTot = data.polarAngles(idxMaxPol);
    maxPolTot = squeeze(maxPolTot(1,:,:));
    maxPol = zeros(1, data.nbFreqs);
    for f = 1:data.nbFreqs
        maxPol(1, f) = maxPolTot(idxMaxAz(1, f), f);
    end
end

maxAz(isnan(di)) = nan;
maxPol(isnan(di)) = nan;
end

%% Interpolate

function [data] = interpolate_directivity_data(data, interp_method, angle_step)
    
    fprintf('\nInterpolate data...\n')

    freq_vec = (1:data.nbFreqs)*data.freqStep;
    nPol = length(data.polarAngles);
    nAzi = length(data.azimuthalAngles);
    
    % duplicate the back direction data
    if ~((180 - data.azimuthalAngles(end)) > 15)
        tmpAmp = nan(nPol, nAzi + 1, data.nbFreqs);
        tmpAmp(:,1:end-1,:) = data.amplitude;
        tmpAmp(:,end,:) = data.amplitude(:,1,:);
        data.amplitude = tmpAmp;
        data.azimuthalAngles = [data.azimuthalAngles 180];
    end

    % if necessary create fine polar angles 
    if nPol > 1
        finePolarAngles = data.polarAngles(1) : angle_step : data.polarAngles(end);
        nPolFine = length(finePolarAngles);
    end
    
    % create fine azimutal angles
    max_azi = data.azimuthalAngles(end);
    fineAzimutalAngles = data.azimuthalAngles(1) : angle_step : max_azi;
    nAziFine = length(fineAzimutalAngles);
    
    if nPol > 1
        tmpAmp = nan(nPolFine, nAziFine, data.nbFreqs);
    
        [pol_grid, az_grid] = meshgrid(data.azimuthalAngles, data.polarAngles);
        
        % loop over frequencies to interpolate amplitude
        for f = 1:data.nbFreqs
            data_to_interpolate = squeeze(data.amplitude(:,:,f));
            idx_not_nan = ~isnan(data_to_interpolate);
            if sum(sum(idx_not_nan)) > 0
                interp_data = griddata(pol_grid(idx_not_nan), az_grid(idx_not_nan), ...
                    data_to_interpolate(idx_not_nan), ...
                    fineAzimutalAngles, finePolarAngles', interp_method);
                if(~isempty(interp_data))
                    tmpAmp(:,:,f) = interp_data;
                end
            end
        end
        data.amplitude = tmpAmp;
        
        % interpolate the background noise
        [polGrid, freqGrid] = meshgrid(freq_vec, data.polarAngles);
        data.bckgdNoise = interp2(polGrid, freqGrid, data.bckgdNoise, ...
        freq_vec, finePolarAngles', interp_method);
    
        data.polarAngles = finePolarAngles;
    else
        % if there is only one measurement horizontal plane
        [azGrid, freqGrid] = meshgrid(freq_vec, data.azimuthalAngles);
        tmpAmp = nan(1, nAziFine, data.nbFreqs);
        tmpAmp(1,:,:) = interp2(azGrid, freqGrid, ...
            squeeze(data.amplitude(1,:,:)), freq_vec, fineAzimutalAngles',...
            interp_method);
        data.amplitude = tmpAmp;
    end
    
    data.azimuthalAngles = fineAzimutalAngles;
end

%% Average on octave or third octave bands

function [data, f_center] = average_bands(data, band_type)

% normalized 1/3 octave center frequencies
f_center = (1000).*((2^(1/3)).^(-20:13));

switch band_type
    case 'octave'
        f_center = f_center(3:3:end);
        a = sqrt(2);
        f_lower_bound = f_center / a;
        f_higher_bound = f_center * a;
        
    case '3d_octave'
        a = sqrt(2^(1/3));	%
        f_lower_bound = f_center / a;
        f_higher_bound = f_center * a;
end
nb_bands = length(f_center);

idx_low = min(data.nbFreqs, round(f_lower_bound / data.freqStep))';
idx_high = min(data.nbFreqs, round(f_higher_bound / data.freqStep))';

nPol = length(data.polarAngles);
nAzi = length(data.azimuthalAngles);

tmp_amp = nan(nPol, nAzi, nb_bands);
tmp_noise = nan(nPol, nb_bands);

% average the amplitude over the bands
for b = 1:nb_bands
    tmp_amp(:,:,b) = mean(data.amplitude(:,:,idx_low(b) : idx_high(b)), 3, 'omitnan');
    tmp_noise(:,b) = mean(data.bckgdNoise(:, idx_low(b) : idx_high(b)), 2, 'omitnan');
end

data.amplitude = tmp_amp;
data.bckgdNoise = tmp_noise;
data.nbFreqs = nb_bands;
end