// ****************************************************************************
// Copyright(C) 2019 by Peter Birkholz, Dresden, Germany
// This file is part of the program MeasureTransferFunction.
// www.vocaltractlab.de
// ****************************************************************************

#include "SpectrumPicture.h"
#include "Data.h"
#include "Dsp.h"


const wxColor SpectrumPicture::SOURCE_SPECTRUM_COLOR = wxColor(180, 180, 180);
const wxColor SpectrumPicture::SOURCE_PRODUCT_SPECTRUM_COLOR = *wxBLACK;
const wxColor SpectrumPicture::REFERENCE_RESPONSE_SPECTRUM_COLOR = *wxRED;
const wxColor SpectrumPicture::PRIMARY_RESPONSE_SPECTRUM_COLOR = *wxBLUE;
const wxColor SpectrumPicture::FINAL_TRANSFER_FUNCTION_COLOR = *wxBLACK;

// ****************************************************************************
// IDs.
// ****************************************************************************

static const int IDM_ZOOM_IN = 1000;
static const int IDM_ZOOM_OUT = 1001;
static const int IDM_SET_AMPLITUDE_RANGE = 1002;

static const int IDM_SHOW_SOURCE_SPECTRUM = 1010;
static const int IDM_SHOW_INVERSE_SOURCE_SPECTRUM = 1011;
static const int IDM_SHOW_SOURCE_PRODUCT_SPECTRUM = 1012;
static const int IDM_SHOW_REFERENCE_RESPONSE_SPECTRUM = 1013;
static const int IDM_SHOW_PRIMARY_RESPONSE_SPECTRUM = 1014;
static const int IDM_SHOW_FINAL_TRANSFER_FUNCTION = 1015;

static const int IDM_SHOW_PHASE = 1020;
static const int IDM_SHOW_FORMANTS = 1021;


// ****************************************************************************
// The event table.
// ****************************************************************************

BEGIN_EVENT_TABLE(SpectrumPicture, BasicPicture)
  EVT_MOUSE_EVENTS(OnMouseEvent)

  EVT_MENU(IDM_ZOOM_IN, OnZoomIn)
  EVT_MENU(IDM_ZOOM_OUT, OnZoomOut)
  EVT_MENU(IDM_SET_AMPLITUDE_RANGE, OnSetAmplitudeRange)

  EVT_MENU(IDM_SHOW_SOURCE_SPECTRUM, OnShowSourceSpectrum)
  EVT_MENU(IDM_SHOW_INVERSE_SOURCE_SPECTRUM, OnShowInverseSourceSpectrum)
  EVT_MENU(IDM_SHOW_SOURCE_PRODUCT_SPECTRUM, OnShowSourceProductSpectrum)
  EVT_MENU(IDM_SHOW_REFERENCE_RESPONSE_SPECTRUM, OnShowReferenceResponseSpectrum)
  EVT_MENU(IDM_SHOW_PRIMARY_RESPONSE_SPECTRUM, OnShowPrimaryResponseSpectrum)
  EVT_MENU(IDM_SHOW_FINAL_TRANSFER_FUNCTION, OnShowFinalTransferFunction)

  EVT_MENU(IDM_SHOW_PHASE, OnShowPhase)
  EVT_MENU(IDM_SHOW_FORMANTS, OnShowFormants)
END_EVENT_TABLE()


// ****************************************************************************
/// Constructor.
// ****************************************************************************

SpectrumPicture::SpectrumPicture(wxWindow *parent) : BasicPicture(parent)
{
  // ****************************************************************
  // Init the variables
  // ****************************************************************

  data = Data::getInstance();
  menuX = 0;
  menuY = 0;

  showSourceSpectrum = false;
  showInverseSourceSpectrum = false;
  showSourceProductSpectrum = false;
  showReferenceResponseSpectrum = true;
  showPrimaryResponseSpectrum = true;
  showFinalTransferFunction = true;

  showPhase = false;
  showFormants = false;

  // ****************************************************************
  // Init the context menu.
  // ****************************************************************

  contextMenu = new wxMenu();

  contextMenu->Append(IDM_ZOOM_IN, "Zoom in (freq. axis)");
  contextMenu->Append(IDM_ZOOM_OUT, "Zoom out (freq. axis)");
  contextMenu->Append(IDM_SET_AMPLITUDE_RANGE, "Set dB range");
  contextMenu->AppendSeparator();

  contextMenu->AppendCheckItem(IDM_SHOW_SOURCE_SPECTRUM, "Show source spectrum");
  contextMenu->AppendCheckItem(IDM_SHOW_INVERSE_SOURCE_SPECTRUM, "Show inverse source spectrum");
  contextMenu->AppendCheckItem(IDM_SHOW_SOURCE_PRODUCT_SPECTRUM, "Show source product spectrum");
  contextMenu->AppendCheckItem(IDM_SHOW_REFERENCE_RESPONSE_SPECTRUM, "Show reference response spectrum");
  contextMenu->AppendCheckItem(IDM_SHOW_PRIMARY_RESPONSE_SPECTRUM, "Show primary response spectrum");
  contextMenu->AppendCheckItem(IDM_SHOW_FINAL_TRANSFER_FUNCTION, "Show final transfer function");
  contextMenu->AppendSeparator();

  contextMenu->AppendCheckItem(IDM_SHOW_PHASE, "Show transfer function phase");
  contextMenu->AppendCheckItem(IDM_SHOW_FORMANTS, "Show transfer function formants");

  // ****************************************************************
  // The spectrum graph
  // ****************************************************************

  graph.init(this, 40, 45, 0, 25);

  graph.initAbscissa(PQ_FREQUENCY, 0.0, 100.0,
    0.0, 0.0, 0.0, 2000.0, 24000.0, 22000.0,
    10, 0, false, false, true);
  graph.initLogOrdinate(1.0, 5.0,
                        -100.0, -1.0, -45.0, 1.0, 100.0, 50.0,
                        true, 10);  // true

  graph.isLinearOrdinate = false;
}


// ****************************************************************************
// ****************************************************************************

void SpectrumPicture::draw(wxDC &dc)
{
  Data *data = Data::getInstance();

  int i;
  int graphX, graphY, graphW, graphH;
  int width, height;

  graph.getDimensions(graphX, graphY, graphW, graphH);
  this->GetSize(&width, &height);


  // ****************************************************************
  // Clear the background and paint the axes.
  // ****************************************************************

  dc.SetBackground(*wxWHITE_BRUSH);
  dc.Clear();
  graph.paintAbscissa(dc);
  graph.paintOrdinate(dc);

  // The scale for the phase angle.

  if (showPhase)
  {
    double y[5];
    wxPen linePen(*wxLIGHT_GREY, 1, wxPENSTYLE_SHORT_DASH);

    dc.SetPen(*wxBLACK_PEN);
    dc.DrawLine(graphX + graphW, graphY, graphX + graphW, graphY + graphH);

    for (i=0; i < 5; i++) 
    { 
      y[i] = graphY + graphH - 1 - 0.25*graphH*i; 

      dc.SetPen(*wxBLACK_PEN);
      dc.DrawLine(graphX + graphW, (int)y[i], graphX + graphW + 6, (int)y[i]);
      
      dc.SetPen(linePen);
      dc.DrawLine(graphX, (int)y[i], graphX + graphW - 1, (int)y[i]);
    }

    dc.SetPen(*wxBLACK_PEN);

    dc.DrawText("Pi",    graphX + graphW + 8, y[4] + 1);
    dc.DrawText("Pi/2",  graphX + graphW + 8, y[3]-8);
    dc.DrawText("0",     graphX + graphW + 8, y[2]-8);
    dc.DrawText("-Pi/2", graphX + graphW + 8, y[1]-8);
    dc.DrawText("-Pi",   graphX + graphW + 8, y[0]-8);

    dc.DrawText("rad",   graphX + graphW + 8, height-16);
  }


  // ****************************************************************
  // Display the selected function.
  // ****************************************************************

  if (showSourceSpectrum)
  {
    paintContinualSpectrum(dc, data->sourceSpectrum, 0.005, SOURCE_SPECTRUM_COLOR, false);
  }

  if (showInverseSourceSpectrum)
  {
    paintContinualSpectrum(dc, data->inverseSourceSpectrum, 0.005, SOURCE_SPECTRUM_COLOR, false);
  }

  if (showSourceProductSpectrum)
  {
    paintContinualSpectrum(dc, data->sourceProductSpectrum, 0.000025, SOURCE_PRODUCT_SPECTRUM_COLOR, false);
  }

  if (showReferenceResponseSpectrum)
  {
    paintContinualSpectrum(dc, data->responseSpectrum[Data::REFERENCE_RESPONSE], 1.0, REFERENCE_RESPONSE_SPECTRUM_COLOR, false);
  }

  if (showPrimaryResponseSpectrum)
  {
    paintContinualSpectrum(dc, data->responseSpectrum[Data::PRIMARY_RESPONSE], 1.0, PRIMARY_RESPONSE_SPECTRUM_COLOR, false);
  }

  if (showFinalTransferFunction)
  {
    paintContinualSpectrum(dc, data->finalTransferFunction, 1.0, FINAL_TRANSFER_FUNCTION_COLOR, showPhase);

    // Calculate and display the formant frequencies.

    if (showFormants)
    {
      const int MAX_FORMANTS = 4;
      /*
      double formantFreq[MAX_FORMANTS];
      double formantBw[MAX_FORMANTS];
      wxString st;
      int i;
      int x, y;

      wxPen dashedPen(*wxBLACK, 1, wxDOT);
      dc.SetPen(dashedPen);

      tlModel->getFormants(formantFreq, formantBw, numFormants, MAX_FORMANTS, frictionNoise, isClosure, isNasal);

      for (i = 0; i < numFormants; i++)
      {
        st = wxString::Format("%d Hz", (int)(formantFreq[i] + 0.5));
        x = graph.getXPos(formantFreq[i]);
        y = 5;
        dc.DrawLine(x, graphY, x, graphY + graphH - 1);
        dc.DrawText(st, x + 5, y);
      }
      */
    }
  }

  // ****************************************************************
  // Print out which curves are shown.
  // ****************************************************************

  dc.SetBackgroundMode(wxSOLID);    // Set a solid white background
  dc.SetTextBackground(wxColor(240, 240, 240));
  wxString st;

  if (showSourceProductSpectrum)
  {
    dc.SetTextForeground(SOURCE_PRODUCT_SPECTRUM_COLOR);
    dc.DrawText("Source product spectrum", graphX + 3, 0);
  }

  if (showReferenceResponseSpectrum)
  {
    dc.SetTextForeground(REFERENCE_RESPONSE_SPECTRUM_COLOR);
    dc.DrawText("Reference response spectrum", graphX + 203, 0);
  }

  if (showPrimaryResponseSpectrum)
  {
    dc.SetTextForeground(PRIMARY_RESPONSE_SPECTRUM_COLOR);
    dc.DrawText("Primary response spectrum", graphX + 403, 0);
  }

  if (showFinalTransferFunction)
  {
    dc.SetTextForeground(FINAL_TRANSFER_FUNCTION_COLOR);
    dc.DrawText("Final transfer function", graphX + 603, 0);
  }
}

// ****************************************************************************
// Paint a continual signal into the spectrum picture.
// ****************************************************************************

void SpectrumPicture::paintContinualSpectrum(wxDC &dc, ComplexSignal *spectrum, double ampFactor, wxColor color, bool showPhase)
{
  const int LINE_WIDTH = 2;   // = 1
  int i;
  int graphX, graphY, graphW, graphH;
  double freq;
  int h0, h1;
  double d0, d1;
  double d;
  int numHarmonics = spectrum->N/2 - 1;
  double F0 = (double)SAMPLING_RATE / (double)spectrum->N;
  int y;
  int lastY;
  wxPen magnitudePen(color, LINE_WIDTH);
  wxPen phasePen(wxColor(0, 0, 255), LINE_WIDTH);

  graph.getDimensions(graphX, graphY, graphW, graphH);

  // ****************************************************************
  // Paint the magnitude spectrum.
  // ****************************************************************

  dc.SetPen(magnitudePen);

  for (i=0; i < graphW; i++)
  {
    freq = graph.getAbsXValue(graphX + i);
    h0 = (int)(freq / F0);
    h1 = h0 + 1;

    // Linear interpolation between the two harmonics
    if ((h0 >= 0) && (h1 < numHarmonics))
    {
      d0 = spectrum->getMagnitude(h0);
      d1 = spectrum->getMagnitude(h1);
      d = d0 + ((d1 - d0)*(freq - h0*F0)) / F0;

      y = graph.getYPos(d*ampFactor);

      if (y < graphY) 
      { 
        y = graphY; 
      }
      if (y >= graphY + graphH) 
      { 
        y = graphY + graphH - 1; 
      }

      if (i > 0) 
      { 
        dc.DrawLine(graphX+i-1, lastY, graphX+i, y); 
      }

      lastY = y;
    }
  }

  // ****************************************************************
  // Paint the phase spectrum.
  // ****************************************************************

  if (showPhase)
  {
    double lastPhase = 0.0;
    double phase;
    int y[2];

    dc.SetPen(phasePen);

    for (i=0; i < graphW; i++)
    {
      freq = graph.getAbsXValue(graphX + i);
      h0 = (int)(freq / F0);
      h1 = h0 + 1;

      // Linear interpolation between the two harmonics
      if ((h0 >= 0) && (h1 < numHarmonics))
      {
        d0 = spectrum->getPhase(h0);
        d1 = spectrum->getPhase(h1);

        if (fabs(d0-d1) > M_PI)
        {
          if (d0 < d1) { d0+= 2.0*M_PI; } else { d0-= 2.0*M_PI; }
        }
        phase = d0 + ((d1 - d0)*(freq - h0*F0)) / F0;

        // When there is a phase jump, paint two separate strokes

        if (fabs(lastPhase - phase) > M_PI)
        {
          // Line rest in the last pixel column
  
          if (i > 0)
          {
            double merkePhase = phase;
            if (phase > lastPhase) { phase-= 2.0*M_PI; } else { phase+= 2.0*M_PI; }

            y[0] = graphY + graphH - 1 - (int)(((double)graphH*(lastPhase + M_PI))/(2.0*M_PI));
            y[1] = graphY + graphH - 1 - (int)(((double)graphH*(phase + M_PI))/(2.0*M_PI));

            if (y[0] < graphY) { y[0] = graphY; }
            if (y[1] < graphY) { y[1] = graphY; }
            if (y[0] >= graphY + graphH) { y[0] = graphY + graphH - 1; }
            if (y[1] >= graphY + graphH) { y[1] = graphY + graphH - 1; }

            dc.DrawLine(graphX+i-1, y[0], graphX+i, y[1]);

            phase = merkePhase;
          }

          if (lastPhase < phase) { lastPhase+= 2.0*M_PI; } else { lastPhase-= 2.0*M_PI; }
        }

        // New line part in the current pixel column

        y[0] = graphY + graphH - 1 - (int)(((double)graphH*(lastPhase + M_PI))/(2.0*M_PI));
        y[1] = graphY + graphH - 1 - (int)(((double)graphH*(phase + M_PI))/(2.0*M_PI));

        if (y[0] < graphY) { y[0] = graphY; }
        if (y[1] < graphY) { y[1] = graphY; }
        if (y[0] >= graphY + graphH) { y[0] = graphY + graphH - 1; }
        if (y[1] >= graphY + graphH) { y[1] = graphY + graphH - 1; }

        if (i > 0)
        { 
          dc.DrawLine(graphX+i-1, y[0], graphX+i, y[1]); 
        }

        lastPhase = phase;
      }
    }
  }

}


// ****************************************************************************
/// Process all mouse events.
// ****************************************************************************

void SpectrumPicture::OnMouseEvent(wxMouseEvent &event)
{
  int mx = event.GetX();
  int my = event.GetY();

  // ****************************************************************
  // The right mouse button changed to down. Call the context menu.
  // ****************************************************************

  if (event.RightDown())
  {
    menuX = mx;
    menuY = my;

    contextMenu->Check(IDM_SHOW_SOURCE_SPECTRUM, showSourceSpectrum);
    contextMenu->Check(IDM_SHOW_INVERSE_SOURCE_SPECTRUM, showInverseSourceSpectrum);
    contextMenu->Check(IDM_SHOW_SOURCE_PRODUCT_SPECTRUM, showSourceProductSpectrum);
    contextMenu->Check(IDM_SHOW_REFERENCE_RESPONSE_SPECTRUM, showReferenceResponseSpectrum);
    contextMenu->Check(IDM_SHOW_PRIMARY_RESPONSE_SPECTRUM, showPrimaryResponseSpectrum);
    contextMenu->Check(IDM_SHOW_FINAL_TRANSFER_FUNCTION, showFinalTransferFunction);
    contextMenu->Check(IDM_SHOW_PHASE, showPhase);
    contextMenu->Check(IDM_SHOW_FORMANTS, showFormants);

    PopupMenu(contextMenu);
    return;
  }
}


// ****************************************************************************
// ****************************************************************************

void SpectrumPicture::OnZoomIn(wxCommandEvent &event)
{
  graph.zoomInAbscissa(false, true);
  this->Refresh();
}


// ****************************************************************************
// ****************************************************************************

void SpectrumPicture::OnZoomOut(wxCommandEvent &event)
{
  graph.zoomOutAbscissa(false, true);
  this->Refresh();
}


// ****************************************************************************
// ****************************************************************************

void SpectrumPicture::OnSetAmplitudeRange(wxCommandEvent &event)
{
  wxString message =
    "Please enter the minimum dB value and the maximum dB value\n"
    "to be displayed, separated by a comma, e.g., \"-40, 40\"";

  wxTextEntryDialog dialog(this, message, "Please enter dB range", "-40, 40", wxOK | wxCANCEL);
  
  if (dialog.ShowModal() == wxID_OK)
  {
    wxString st = dialog.GetValue();

    int commaPos = st.Find(',');
    if (commaPos == wxNOT_FOUND)
    {
      wxMessageBox("No comma found!", "Error");
      return;
    }

    wxString firstString = st.Mid(0, commaPos);
    wxString secondString = st.Mid(commaPos + 1);

    double lowerLevel = 0.0;
    double upperLevel = 0.0;

    bool lowerLevelOk = firstString.ToDouble(&lowerLevel);
    bool upperLevelOk = secondString.ToDouble(&upperLevel);

    if ((!lowerLevelOk) || (!upperLevelOk))
    {
      wxMessageBox("Both values before and after the comma must be valid numbers!", "Error");
      return;
    }

    if (lowerLevel >= upperLevel)
    {
      wxMessageBox("The lower limit must be below the upper limit!", "Error");
      return;
    }

    if (lowerLevel < graph.logOrdinate.lowerLevelMin)
    {
      lowerLevel = graph.logOrdinate.lowerLevelMin;
    }
    if (lowerLevel > graph.logOrdinate.lowerLevelMax)
    {
      lowerLevel = graph.logOrdinate.lowerLevelMax;
    }

    if (upperLevel < graph.logOrdinate.upperLevelMin)
    {
      upperLevel = graph.logOrdinate.upperLevelMin;
    }
    if (upperLevel > graph.logOrdinate.upperLevelMax)
    {
      upperLevel = graph.logOrdinate.upperLevelMax;
    }

    graph.logOrdinate.lowerLevel = lowerLevel;
    graph.logOrdinate.upperLevel = upperLevel;
    
    this->Refresh();
  }
}


// ****************************************************************************
// ****************************************************************************

void SpectrumPicture::OnShowSourceSpectrum(wxCommandEvent &event)
{
  showSourceSpectrum = !showSourceSpectrum;
  this->Refresh();
}


// ****************************************************************************
// ****************************************************************************

void SpectrumPicture::OnShowInverseSourceSpectrum(wxCommandEvent &event)
{
  showInverseSourceSpectrum = !showInverseSourceSpectrum;
  this->Refresh();
}


// ****************************************************************************
// ****************************************************************************

void SpectrumPicture::OnShowSourceProductSpectrum(wxCommandEvent &event)
{
  showSourceProductSpectrum = !showSourceProductSpectrum;
  this->Refresh();
}


// ****************************************************************************
// ****************************************************************************

void SpectrumPicture::OnShowReferenceResponseSpectrum(wxCommandEvent &event)
{
  showReferenceResponseSpectrum = !showReferenceResponseSpectrum;
  this->Refresh();
}


// ****************************************************************************
// ****************************************************************************

void SpectrumPicture::OnShowPrimaryResponseSpectrum(wxCommandEvent &event)
{
  showPrimaryResponseSpectrum = !showPrimaryResponseSpectrum;
  this->Refresh();
}


// ****************************************************************************
// ****************************************************************************

void SpectrumPicture::OnShowFinalTransferFunction(wxCommandEvent &event)
{
  showFinalTransferFunction = !showFinalTransferFunction;
  this->Refresh();
}


// ****************************************************************************
// ****************************************************************************

void SpectrumPicture::OnShowPhase(wxCommandEvent &event)
{
  showPhase = !showPhase;
  this->Refresh();
}


// ****************************************************************************
// ****************************************************************************

void SpectrumPicture::OnShowFormants(wxCommandEvent &event)
{
  wxMessageBox("This function has not been implemented.", "Information");

//  showFormants = !showFormants;
  this->Refresh();
}

// ****************************************************************************

