// ****************************************************************************
// Copyright(C) 2019 by Peter Birkholz, Dresden, Germany
// This file is part of the program MeasureTransferFunction.
// www.vocaltractlab.de
// ****************************************************************************

#include "SignalPicture.h"

const wxColor SignalPicture::SOURCE_SIGNAL_COLOR = *wxLIGHT_GREY;
const wxColor SignalPicture::SWEEP_RESPONSE_COLOR[Data::NUM_RESPONSE_SIGNALS] = 
{ *wxRED, *wxBLUE };


// ****************************************************************************
// IDs.
// ****************************************************************************

static const int IDM_ZOOM_IN = 1000;
static const int IDM_ZOOM_OUT = 1001;
static const int IDM_SHOW_SOURCE_SIGNAL = 1020;

// ****************************************************************************
// The event table.
// ****************************************************************************

BEGIN_EVENT_TABLE(SignalPicture, BasicPicture)
  EVT_MOUSE_EVENTS(OnMouseEvent)

  EVT_MENU(IDM_ZOOM_IN, OnZoomIn)
  EVT_MENU(IDM_ZOOM_OUT, OnZoomOut)
  EVT_MENU(IDM_SHOW_SOURCE_SIGNAL, OnShowSourceSignal)
END_EVENT_TABLE()


// ****************************************************************************
/// Construcor. Passes the parent parameter.
// ****************************************************************************

SignalPicture::SignalPicture(wxWindow *parent) : BasicPicture(parent)
{
  // The time axis.

  graph.init(this, 40, 0, 0, 25);

  graph.initAbscissa(PQ_TIME, 0.0, 0.001,
    0.0, 0.0, 0.0, 0.01, Data::TRACK_LENGTH_S, 5.0,
    10, 3, false, false, true);

  graph.initLinearOrdinate(PQ_NONE, 0.0, 0.1,
    -1.0, -1.0, -1.0, 1.0, 1.0, 1.0,
    1, 1, false, false, true);

  // Init some variables.
  
  data = Data::getInstance();
  menuX = 0;
  menuY = 0;
  lastMx = 0;
  lastMy = 0;

  showSourceSignal = true;

  // Init the context menu.
  
  contextMenu = new wxMenu();
  contextMenu->Append(IDM_ZOOM_IN, "Zoom in");
  contextMenu->Append(IDM_ZOOM_OUT, "Zoom out");
  contextMenu->AppendSeparator();

  contextMenu->AppendCheckItem(IDM_SHOW_SOURCE_SIGNAL, "Show source signal");
}


// ****************************************************************************
// ****************************************************************************

void SignalPicture::draw(wxDC &dc)
{
  paintOscillograms(dc);
}


// ****************************************************************************
// ****************************************************************************

void SignalPicture::paintOscillograms(wxDC &dc)
{
  Data *data = Data::getInstance();

  int graphX, graphY, graphW, graphH;
  int width, height;

  graph.getDimensions(graphX, graphY, graphW, graphH);
  this->GetSize(&width, &height);


  // ****************************************************************
  // Clear the background and paint the axes.
  // ****************************************************************

  dc.SetBackground(*wxWHITE_BRUSH);
  dc.Clear();
  graph.paintAbscissa(dc);
  graph.paintOrdinate(dc);


  // ****************************************************************
  // Draw the signals.
  // ****************************************************************

  if (showSourceSignal)
  {
    paintOscillogram(dc, data->sourceSignal, SOURCE_SIGNAL_COLOR);
  }

  paintOscillogram(dc, data->sweepResponse[data->selectedResponse], SWEEP_RESPONSE_COLOR[data->selectedResponse]);


  // ****************************************************************
  // Output some text.
  // ****************************************************************

  dc.SetPen(*wxBLACK_PEN);
  dc.SetBackgroundMode(wxSOLID);    // Set a solid white background
  dc.SetTextBackground(wxColor(240, 240, 240));
  wxString st;

  st = wxString::Format("View range: %2.3f s",
    graph.abscissa.positiveLimit);
  dc.DrawText(st, graphX + 3, 0);

  if (showSourceSignal)
  {
    dc.SetTextForeground(SOURCE_SIGNAL_COLOR);
    dc.DrawText("Source signal", graphX + 3, 40);

    st = wxString::Format("Source sweep freq = [%2.0f, %2.0f, %2.0f, %2.0f]   Source amp = %2.2f",
      data->currSweepFreq_Hz[0], data->currSweepFreq_Hz[1],
      data->currSweepFreq_Hz[2], data->currSweepFreq_Hz[3], data->sourceAmpFactor);
    dc.DrawText(st, graphX + 150, 0);
  }

  dc.SetTextForeground(SWEEP_RESPONSE_COLOR[data->selectedResponse]);
  if (data->selectedResponse == Data::PRIMARY_RESPONSE)
  {
    dc.DrawText("Primary sweep response", graphX + 3, 20);
  }
  else
  {
    dc.DrawText("Reference sweep response", graphX + 3, 20);
  }

  dc.SetPen(*wxBLACK_PEN);
}


// ****************************************************************************
// Draw a single oscillogram.
// ****************************************************************************

void SignalPicture::paintOscillogram(wxDC &dc, Signal *s, const wxColor &color)
{
  int i, k;
  int graphX, graphY, graphW, graphH;
  double tLeft_s, tRight_s;
  int sampleIndexLeft, sampleIndexRight;
  double lastSampleValue = 0.0;
  double minValue, maxValue;
  int minY, maxY;
  int stepSize;

  graph.getDimensions(graphX, graphY, graphW, graphH);

  dc.SetPen(wxPen(color));

  // ****************************************************************
  // Run through all pixels from left to right.
  // ****************************************************************

  for (i = 0; i < graphW; i++)
  {
    // Time and audio sampling index at the left and right edge of a pixel.
    tLeft_s = graph.getAbsXValue(graphX + i);
    tRight_s = graph.getAbsXValue(graphX + i + 1);
    sampleIndexLeft = (int)(tLeft_s * SAMPLING_RATE);
    sampleIndexRight = (int)(tRight_s * SAMPLING_RATE);

    if ((sampleIndexLeft >= 0) && (sampleIndexRight < s->N))
    {
      minValue = lastSampleValue;
      maxValue = lastSampleValue;

      // Increase the step size for the loop if there are more than 
      // 32 signal samples in the range of a single pixel.
      stepSize = 1;
      if (sampleIndexRight - sampleIndexLeft > 32)
      {
        stepSize = 1 + (sampleIndexRight - sampleIndexLeft) / 32;
      }
      
      for (k = sampleIndexLeft; k <= sampleIndexRight; k+= stepSize)
      {
        if (s->x[k] < minValue)
        {
          minValue = s->x[k];
        }
        if (s->x[k] > maxValue)
        {
          maxValue = s->x[k];
        }
      }
      
      k = sampleIndexRight;
      if (s->x[k] < minValue)
      {
        minValue = s->x[k];
      }
      if (s->x[k] > maxValue)
      {
        maxValue = s->x[k];
      }

      lastSampleValue = s->x[sampleIndexRight];

      minY = graph.getYPos(maxValue);
      maxY = graph.getYPos(minValue);

      if (minY < graphY)
      {
        minY = graphY;
      }
      if (minY >= graphY + graphH)
      {
        minY = graphY + graphH - 1;
      }

      if (maxY < graphY)
      {
        maxY = graphY;
      }
      if (maxY >= graphY + graphH)
      {
        maxY = graphY + graphH - 1;
      }

      if (minY == maxY)
      {
        dc.DrawPoint(graphX + i, minY);
      }
      else
      {
        dc.DrawLine(graphX + i, minY, graphX + i, maxY);
      }
    }
  }
}


// ****************************************************************************
/// Process all mouse events.
// ****************************************************************************

void SignalPicture::OnMouseEvent(wxMouseEvent &event)
{
  int width, height;
  this->GetSize(&width, &height);

  int mx = event.GetX();
  int my = event.GetY();

  // ****************************************************************
  // The right mouse button changed to down. Call the context menu.
  // ****************************************************************

  if (event.RightDown())
  {
    menuX = mx;
    menuY = my;
    
    contextMenu->Check(IDM_SHOW_SOURCE_SIGNAL, showSourceSignal);
    PopupMenu(contextMenu);
    return;
  }

}


// ****************************************************************************
// ****************************************************************************

void SignalPicture::OnZoomIn(wxCommandEvent &event)
{
  graph.zoomInAbscissa(false, true);
  this->Refresh();
}


// ****************************************************************************
// ****************************************************************************

void SignalPicture::OnZoomOut(wxCommandEvent &event)
{
  graph.zoomOutAbscissa(false, true);
  this->Refresh();
}


// ****************************************************************************
// ****************************************************************************

void SignalPicture::OnShowSourceSignal(wxCommandEvent &event)
{
  showSourceSignal = !showSourceSignal;
  this->Refresh();
}


// ****************************************************************************

