// ****************************************************************************
// Copyright(C) 2019 by Peter Birkholz, Dresden, Germany
// This file is part of the program MeasureTransferFunction.
// www.vocaltractlab.de
// ****************************************************************************

#include "MainWindow.h"

#include <wx/statline.h>
#include <wx/spinctrl.h>
#include <iostream>
#include "Data.h"
#include "Dsp.h"
#include "SoundLib.h"
#include "AudioFile.h"
#include "SilentMessageBox.h"

using namespace std;

// ****************************************************************************
// IDs.
// ****************************************************************************

// Menu
static const int IDM_TEST = 900;
static const int IDM_RESET = 901;

// Left side controls
static const int IDB_UPDATE_DEVICE_LISTS = 1000;
static const int IDL_AUDIO_OUTPUT_DEVICE = 1001;
static const int IDL_AUDIO_INPUT_DEVICE = 1002;
static const int IDB_SET_DEVICES = 1003;

static const int TXT_FADE_IN_LOWER_FREQ = 1010;
static const int TXT_FADE_IN_UPPER_FREQ = 1011;
static const int TXT_FADE_OUT_LOWER_FREQ = 1012;
static const int TXT_FADE_OUT_UPPER_FREQ = 1013;

static const int TXT_SOURCE_AMP_FACTOR = 1014;
static const int IDB_GENERATE_SWEEP = 1015;
static const int IDB_PLAY_SOURCE_SIGNAL = 1016;

static const int IDR_SIGNAL_SELECTION = 1050;

static const int IDB_RECORD = 1060;
static const int IDB_REPLAY = 1061;
static const int IDB_SAVE_SIGNAL = 1062;
static const int IDB_LOAD_SIGNAL = 1063;
static const int IDB_SAVE_SPECTRUM = 1066;

static const int IDB_CALC_FINAL_TRANSFER_FUNCTION = 1030;
static const int IDB_SAVE_FINAL_TRANSFER_FUNCTION = 1031;

// Right side controls
static const int IDS_TIME1 = 1100;
static const int IDS_TIME2 = 1101;

// Keys
static const int IDK_CTRL_LEFT              = 1300;
static const int IDK_CTRL_RIGHT             = 1301;


// ****************************************************************************
// The event table.
// ****************************************************************************

BEGIN_EVENT_TABLE(MainWindow, wxFrame)
  EVT_COMMAND(wxID_ANY, updateRequestEvent, OnUpdateRequest)
  EVT_CLOSE(OnCloseWindow)

  // Controls on the left side
  EVT_BUTTON(IDB_UPDATE_DEVICE_LISTS, OnUpdateDeviceLists)
  EVT_BUTTON(IDB_SET_DEVICES, OnSetDevices)

  EVT_BUTTON(IDB_GENERATE_SWEEP, OnGenerateSweep)
  EVT_BUTTON(IDB_PLAY_SOURCE_SIGNAL, OnPlaySourceSignal)

  EVT_RADIOBOX(IDR_SIGNAL_SELECTION, OnSignalSelection)

  EVT_BUTTON(IDB_RECORD, OnRecord)
  EVT_BUTTON(IDB_REPLAY, OnReplay)
  EVT_BUTTON(IDB_SAVE_SIGNAL, OnSaveSignal)
  EVT_BUTTON(IDB_LOAD_SIGNAL, OnLoadSignal)
  EVT_BUTTON(IDB_SAVE_SPECTRUM, OnSaveSpectrum)

  EVT_BUTTON(IDB_CALC_FINAL_TRANSFER_FUNCTION, OnCalcFinalTransferFunction)
  EVT_BUTTON(IDB_SAVE_FINAL_TRANSFER_FUNCTION, OnSaveFinalTransferFunction)

  EVT_COMMAND_SCROLL(IDS_TIME1, OnScrollTime1)
  EVT_COMMAND_SCROLL(IDS_TIME2, OnScrollTime2)

  // Menu events
  EVT_MENU(IDM_TEST, OnTest)
  EVT_MENU(IDM_RESET, OnReset)
  EVT_MENU(wxID_EXIT, OnExit)
  EVT_MENU(wxID_ABOUT, OnAbout)

  // Key events
  EVT_MENU(IDK_CTRL_LEFT, OnKeyCtrlLeft)
  EVT_MENU(IDK_CTRL_RIGHT, OnKeyCtrlRight)

END_EVENT_TABLE()


// ****************************************************************************
// ****************************************************************************

MainWindow::MainWindow(wxWindow *parent, wxWindowID id, const wxString &title)
{

  // ****************************************************************
  // Start PortAudio.
  // ****************************************************************

  initSound(SAMPLING_RATE, WAVE_MAPPER, WAVE_MAPPER);

  // ****************************************************************
  // Init the variables BEFORE the child widgets.
  // ****************************************************************

  data = Data::getInstance();
  
  // ****************************************************************
  // ****************************************************************

	wxFrame::Create(parent, id, title, wxDefaultPosition, 
    wxDefaultSize, wxCLOSE_BOX | wxMINIMIZE_BOX | 
    wxMAXIMIZE_BOX | wxSYSTEM_MENU | wxCAPTION | 
    wxRAISED_BORDER | wxRESIZE_BORDER);

  initWidgets();
  updateWidgets();
  
  updateDeviceList();

  // Make the main window double buffered to avoid any flickering
  // of the child-windows and during resizing.
  this->SetDoubleBuffered(true);
}


// ****************************************************************************
// ****************************************************************************

void MainWindow::initWidgets()
{
  wxString st;

  // Make the background color the same as that for buttons.
  this->SetBackgroundColour(wxSystemSettings::GetColour(wxSYS_COLOUR_BTNFACE));

  // ****************************************************************
  // Accellerator keys.
  // ****************************************************************

  const int NUM_ACCELS = 2;
  wxAcceleratorEntry entries[NUM_ACCELS];
  entries[0].Set(wxACCEL_CTRL, WXK_LEFT,  IDK_CTRL_LEFT);
  entries[1].Set(wxACCEL_CTRL, WXK_RIGHT, IDK_CTRL_RIGHT);

  wxAcceleratorTable accel(NUM_ACCELS, entries);
  this->SetAcceleratorTable(accel);


  // ****************************************************************
  // Set properties of this window.
  // ****************************************************************

  this->SetSize(20, 20, 900, 700);

  // ****************************************************************
  // Create the menu.
  // ****************************************************************

  wxMenu *menu = NULL;
  
  menuBar = new wxMenuBar();

  menu = new wxMenu();
  menu->Append(IDM_RESET, "Reset all");
  menu->Append(wxID_EXIT, "Exit");

  menuBar->Append(menu, "File");

  // ****************************************************************

  menu = new wxMenu();
  menu->Append(wxID_ABOUT, "About");

  menuBar->Append(menu, "Help");

  // ****************************************************************

  this->SetMenuBar(menuBar);

  // ****************************************************************
  // Create the actual widgets.
  // ****************************************************************

  wxButton *button = NULL;
  wxStaticText *label = NULL;
  wxBoxSizer *topLevelSizer = new wxBoxSizer(wxHORIZONTAL);
  wxBoxSizer *sizer = NULL;

  // ****************************************************************
  // Left side sizer.
  // ****************************************************************

  wxBoxSizer *leftSizer = new wxBoxSizer(wxVERTICAL);

  leftSizer->AddSpacer(5);

  button = new wxButton(this, IDB_UPDATE_DEVICE_LISTS, "Update device lists");
  leftSizer->Add(button, 0, wxALL | wxGROW, 3);

  label = new wxStaticText(this, wxID_ANY, "Audio output device:");
  leftSizer->Add(label, 0, wxALL | wxGROW, 3);

  lstAudioOutputDevice = new wxComboBox(this, IDL_AUDIO_OUTPUT_DEVICE, "", wxDefaultPosition,
    wxDefaultSize, 0, NULL, wxCB_READONLY);
  leftSizer->Add(lstAudioOutputDevice, 0, wxALL | wxGROW, 3);

  label = new wxStaticText(this, wxID_ANY, "Audio input device:");
  leftSizer->Add(label, 0, wxALL | wxGROW, 3);

  lstAudioInputDevice = new wxComboBox(this, IDL_AUDIO_INPUT_DEVICE, "", wxDefaultPosition,
    wxDefaultSize, 0, NULL, wxCB_READONLY);
  leftSizer->Add(lstAudioInputDevice, 0, wxALL | wxGROW, 3);

  button = new wxButton(this, IDB_SET_DEVICES, "Set devices");
  leftSizer->Add(button, 0, wxALL | wxGROW, 3);

  leftSizer->AddSpacer(20);

  // ****************************************************************
  // Fade-in frequencies.
  // ****************************************************************

  sizer = new wxBoxSizer(wxHORIZONTAL);

  label = new wxStaticText(this, wxID_ANY, "Fade in from   ");
  sizer->Add(label, 0, wxALL | wxGROW, 3);

  txtFadeInLowerFreq = new wxTextCtrl(this, TXT_FADE_IN_LOWER_FREQ, 
    wxString::Format("%2.0f", data->currSweepFreq_Hz[0]), wxDefaultPosition, wxSize(60, -1));
  sizer->Add(txtFadeInLowerFreq, 0, wxALL | wxGROW, 3);

  label = new wxStaticText(this, wxID_ANY, "to");
  sizer->Add(label, 0, wxALL | wxGROW, 3);

  txtFadeInUpperFreq = new wxTextCtrl(this, TXT_FADE_IN_UPPER_FREQ, 
    wxString::Format("%2.0f", data->currSweepFreq_Hz[1]), wxDefaultPosition, wxSize(60, -1));
  sizer->Add(txtFadeInUpperFreq, 0, wxALL | wxGROW, 3);

  label = new wxStaticText(this, wxID_ANY, "Hz");
  sizer->Add(label, 0, wxALL | wxGROW, 3);

  leftSizer->Add(sizer, 0, wxGROW);

  // ****************************************************************
  // Fade-out frequencies.
  // ****************************************************************

  sizer = new wxBoxSizer(wxHORIZONTAL);

  label = new wxStaticText(this, wxID_ANY, "Fade out from");
  sizer->Add(label, 0, wxALL | wxGROW, 3);

  txtFadeOutLowerFreq = new wxTextCtrl(this, TXT_FADE_OUT_LOWER_FREQ, 
    wxString::Format("%2.0f", data->currSweepFreq_Hz[2]), wxDefaultPosition, wxSize(60, -1));
  sizer->Add(txtFadeOutLowerFreq, 0, wxALL | wxGROW, 3);

  label = new wxStaticText(this, wxID_ANY, "to");
  sizer->Add(label, 0, wxALL | wxGROW, 3);

  txtFadeOutUpperFreq = new wxTextCtrl(this, TXT_FADE_OUT_UPPER_FREQ, 
    wxString::Format("%2.0f", data->currSweepFreq_Hz[3]), wxDefaultPosition, wxSize(60, -1));
  sizer->Add(txtFadeOutUpperFreq, 0, wxALL | wxGROW, 3);

  label = new wxStaticText(this, wxID_ANY, "Hz");
  sizer->Add(label, 0, wxALL | wxGROW, 3);
  
  leftSizer->Add(sizer, 0, wxGROW);
  

  // ****************************************************************

  sizer = new wxBoxSizer(wxHORIZONTAL);

  label = new wxStaticText(this, wxID_ANY, "Source amplitude (0.0...1.0):");
  sizer->Add(label, 0, wxALL | wxGROW, 3);

  st = wxString::Format("%2.2f", data->sourceAmpFactor);

  txtSourceAmpFactor = new wxTextCtrl(this, TXT_SOURCE_AMP_FACTOR, st, 
    wxDefaultPosition, wxSize(80, -1));
  sizer->Add(txtSourceAmpFactor, 0, wxALL | wxGROW, 3);

  leftSizer->Add(sizer, 0, wxGROW);

  // ****************************************************************

  button = new wxButton(this, IDB_GENERATE_SWEEP, "Generate source sweep");
  leftSizer->Add(button, 0, wxALL | wxGROW, 3);

  button = new wxButton(this, IDB_PLAY_SOURCE_SIGNAL, "Play source signal");
  leftSizer->Add(button, 0, wxALL | wxGROW, 3);

  leftSizer->AddSpacer(20);

  // ****************************************************************

  wxArrayString choices;
  choices.Add("Reference response");
  choices.Add("Primary response");

  radSignalSelection = new wxRadioBox(this, IDR_SIGNAL_SELECTION, "Selected response", wxDefaultPosition,
    wxDefaultSize, choices, 1);

  leftSizer->Add(radSignalSelection, 0, wxALL | wxGROW, 3);

  // ****************************************************************

  button = new wxButton(this, IDB_RECORD, "Record");
  leftSizer->Add(button, 0, wxALL | wxGROW, 3);

  button = new wxButton(this, IDB_REPLAY, "Replay");
  leftSizer->Add(button, 0, wxALL | wxGROW, 3);

  button = new wxButton(this, IDB_SAVE_SIGNAL, "Save response waveform");
  leftSizer->Add(button, 0, wxALL | wxGROW, 3);

  button = new wxButton(this, IDB_LOAD_SIGNAL, "Load response waveform");
  leftSizer->Add(button, 0, wxALL | wxGROW, 3);

  button = new wxButton(this, IDB_SAVE_SPECTRUM, "Export response spectrum");
  leftSizer->Add(button, 0, wxALL | wxGROW, 3);

  leftSizer->AddSpacer(20);

  // ****************************************************************

  button = new wxButton(this, IDB_CALC_FINAL_TRANSFER_FUNCTION, "Calc. final transfer function");
  leftSizer->Add(button, 0, wxALL | wxGROW, 3);

  button = new wxButton(this, IDB_SAVE_FINAL_TRANSFER_FUNCTION, "Export final transfer function");
  leftSizer->Add(button, 0, wxALL | wxGROW, 3);

  // ****************************************************************

  // Add the left sizer to the top-level sizer!
  topLevelSizer->Add(leftSizer, 0, wxGROW | wxALL, 5);

  // ****************************************************************
  // Static line to separate the left and right part.
  // ****************************************************************

  wxStaticLine *verticalLine = new wxStaticLine(this, wxID_ANY, wxDefaultPosition,
    wxDefaultSize, wxLI_VERTICAL);

  topLevelSizer->Add(verticalLine, 0, wxGROW | wxALL, 2);

  // ****************************************************************
  // The right sizer with the big pictures.
  // ****************************************************************

  wxBoxSizer *rightSizer = new wxBoxSizer(wxVERTICAL);

  picSignal = new SignalPicture(this);
  rightSizer->Add(picSignal, 1, wxGROW);

  scrTime1 = new wxScrollBar(this, IDS_TIME1);
  scrTime1->SetScrollbar(0, 100, Data::TRACK_LENGTH_S * 100, 100);
  rightSizer->Add(scrTime1, 0, wxGROW);

  picImpulseResponse = new ImpulseResponsePicture(this);
  rightSizer->Add(picImpulseResponse, 1, wxGROW);

  scrTime2 = new wxScrollBar(this, IDS_TIME2);
  scrTime2->SetScrollbar(0, 100, Data::TRACK_LENGTH_S * 100, 100);
  rightSizer->Add(scrTime2, 0, wxGROW);

  picSpectrum = new SpectrumPicture(this);
  rightSizer->Add(picSpectrum, 1, wxGROW);

  // Add the right sizer to the top-level sizer!
  topLevelSizer->Add(rightSizer, 1, wxGROW | wxALL, 5);

  // ****************************************************************
  // Set the top-level sizer for this window.
  // ****************************************************************

  this->SetSizer(topLevelSizer);
}


// ****************************************************************************
// ****************************************************************************

void MainWindow::updateWidgets()
{
  wxString st;

  // ****************************************************************
  // Two time scroll bars.
  // ****************************************************************

  double pos_ticks = picSignal->graph.abscissa.reference * 100.0;
  int pos = (int)(pos_ticks + 0.5);
  if (pos != scrTime1->GetThumbPosition())
  {
    scrTime1->SetThumbPosition(pos);
  }

  pos_ticks = picImpulseResponse->graph.abscissa.reference * 100.0;
  pos = (int)(pos_ticks + 0.5);
  if (pos != scrTime2->GetThumbPosition())
  {
    scrTime2->SetThumbPosition(pos);
  }

  // ****************************************************************
  // Update the parameters of the source sweep.
  // ****************************************************************

  st = wxString::Format("%2.0f", data->currSweepFreq_Hz[0]);
  txtFadeInLowerFreq->SetValue(st);

  st = wxString::Format("%2.0f", data->currSweepFreq_Hz[1]);
  txtFadeInUpperFreq->SetValue(st);

  st = wxString::Format("%2.0f", data->currSweepFreq_Hz[2]);
  txtFadeOutLowerFreq->SetValue(st);

  st = wxString::Format("%2.0f", data->currSweepFreq_Hz[3]);
  txtFadeOutUpperFreq->SetValue(st);

  st = wxString::Format("%2.2f", data->sourceAmpFactor);
  txtSourceAmpFactor->SetValue(st);

  // ****************************************************************
  // Signal selection.
  // ****************************************************************

  radSignalSelection->SetSelection((int)data->selectedResponse);
}


// ****************************************************************************
/// Refill the combo boxes with the audio devices.
// ****************************************************************************

void MainWindow::updateDeviceList()
{
  int i;
  WAVEINCAPS waveInCaps;
  WAVEOUTCAPS waveOutCaps;

  // ****************************************************************
  // Input devices
  // ****************************************************************

  lstAudioInputDevice->Clear();
  int numInputDevices = waveInGetNumDevs();
  for (i = 0; i < numInputDevices; i++)
  {
    waveInGetDevCaps(i, &waveInCaps, sizeof(waveInCaps));
    lstAudioInputDevice->Append(wxString(waveInCaps.szPname));
  }

  // ****************************************************************
  // Output devices
  // ****************************************************************

  lstAudioOutputDevice->Clear();
  int numOutputDevices = waveOutGetNumDevs();
  for (i = 0; i < numOutputDevices; i++)
  {
    waveOutGetDevCaps(i, &waveOutCaps, sizeof(waveOutCaps));
    lstAudioOutputDevice->Append(wxString(waveOutCaps.szPname));
  }
}


// ****************************************************************************
/// Play the given signal.
// ****************************************************************************

void MainWindow::playSignal(Signal *s)
{
  int i;
  TemplateSignal<float> floatSignal(s->N);

  // ****************************************************************
  // Convert the signal from double to float.
  // ****************************************************************

  for (i = 0; i < s->N; i++)
  {
    floatSignal.x[i] = (float)s->x[i];
  }

  if (waveStartPlaying(floatSignal.x, floatSignal.N) == false)
  {
    wxMessageBox("Playing failed", "Error");
    return;
  }

  // ****************************************************************
  // Wait until the user stops playing.
  // ****************************************************************

  SilentMessageBox dialog("Press OK to stop playing", "Stop playing?", this);
  dialog.ShowModal();

  waveStopPlaying();
}


// ****************************************************************************
/// An update request received from a child widget.
// ****************************************************************************

void MainWindow::OnUpdateRequest(wxCommandEvent &event)
{
  updateWidgets();
}


// ****************************************************************************
// ****************************************************************************

void MainWindow::OnCloseWindow(wxCloseEvent &event)
{
  if (wxMessageBox("Do you really want to quit?", "Quit", wxYES_NO, this) == wxYES)
  {
    // Close the audio devices.
    exitSound();

    this->Destroy();
    exit(0);
  }
}


// ****************************************************************************
// ****************************************************************************

void MainWindow::OnTest(wxCommandEvent &event)
{
  // Artificially create the recorded "direct" signal by
  // convolution of the source signal with a short exponential decay.

  const int NUM_FILTER_SAMPLES = 400;
  const double FACTOR = 0.99;
  double filter[NUM_FILTER_SAMPLES];
  int i, k;

  // Create the filter.
  
  filter[0] = 0.2;
  for (i = 1; i < NUM_FILTER_SAMPLES; i++)
  {
    filter[i] = filter[i - 1] * FACTOR;
  }

  // Do the convolution.
  
  Signal *s = data->sweepResponse[Data::REFERENCE_RESPONSE];
  s->setZero();
  double sum = 0.0;

  for (i = NUM_FILTER_SAMPLES; i < s->N; i++)
  {
    sum = 0.0;
    for (k = 0; k < NUM_FILTER_SAMPLES; k++)
    {
      sum += filter[k] * data->sourceSignal->x[i - k];
    }
    s->x[i] = sum;
  }

  picSignal->Refresh();
  picSpectrum->Refresh();
}


// ****************************************************************************
// ****************************************************************************

void MainWindow::OnReset(wxCommandEvent &event)
{
  wxString st = "Are you sure you want to clear all signals and\n"
    "reset all settings?";

  if (wxMessageBox(st, "Are you sure?", wxYES_NO, this) == wxYES)
  {
    // Clear all signals.

    data->sourceSignal->setZero();
    data->inverseSourceSignal->setZero();

    data->sourceSpectrum->setToOne();
    data->inverseSourceSpectrum->setToOne();
    data->sourceProductSpectrum->setToOne();
    data->finalTransferFunction->setToOne();

    for (int i = 0; i < Data::NUM_RESPONSE_SIGNALS; i++)
    {
      data->sweepResponse[i]->setZero();
      data->impulseResponse[i]->setZero();
      data->responseSpectrum[i]->setToOne();
      data->peakNormalizationFactor[i] = 1.0;
    }

    // Reset all parameters.

    data->selectionMark_s[0] = -1.0;
    data->selectionMark_s[1] = -1.0;
    data->mark_s = 0.0;
    data->sourceAmpFactor = 0.5;

    data->currSweepFreq_Hz[0] = Data::DEFAULT_FADE_IN_LOWER_FREQ;     // Start of fade-in
    data->currSweepFreq_Hz[1] = Data::DEFAULT_FADE_IN_UPPER_FREQ;     // End of fade-in
    data->currSweepFreq_Hz[2] = Data::DEFAULT_FADE_OUT_LOWER_FREQ;    // Start of fade-out
    data->currSweepFreq_Hz[3] = Data::DEFAULT_FADE_OUT_UPPER_FREQ;    // End of fade-out

    // Generate the source sweep.
    data->createExponentialSweep(data->currSweepFreq_Hz);

    updateWidgets();

    picSignal->Refresh();
    picImpulseResponse->Refresh();
    picSpectrum->Refresh();
  }
}


// ****************************************************************************
// ****************************************************************************

void MainWindow::OnExit(wxCommandEvent &event)
{
  Close(true);
}

// ****************************************************************************
// ****************************************************************************

void MainWindow::OnAbout(wxCommandEvent &event)
{
  wxMessageDialog dialog(this,
    "This software can measure transfer functions. Created by Peter Birkholz since Feb. 2015.");
  dialog.ShowModal();
}


// ****************************************************************************
// ****************************************************************************

void MainWindow::OnKeyCtrlLeft(wxCommandEvent &event)
{
  picImpulseResponse->graph.abscissa.reference -= 0.02;   // = 20 ms
  if (picImpulseResponse->graph.abscissa.reference < 0.0)
  {
    picImpulseResponse->graph.abscissa.reference = 0.0;
  }

  updateWidgets();

  picSignal->Refresh();
  picImpulseResponse->Refresh();
}

// ****************************************************************************
// ****************************************************************************

void MainWindow::OnKeyCtrlRight(wxCommandEvent &event)
{
  picImpulseResponse->graph.abscissa.reference += 0.02;   // = 20 ms
  if (picImpulseResponse->graph.abscissa.reference > Data::TRACK_LENGTH_S)
  {
    picImpulseResponse->graph.abscissa.reference = Data::TRACK_LENGTH_S;
  }

  updateWidgets();
  
  picSignal->Refresh();
  picImpulseResponse->Refresh();
}


// ****************************************************************************
// ****************************************************************************

void MainWindow::OnUpdateDeviceLists(wxCommandEvent& event)
{
  updateDeviceList();
}


// ****************************************************************************
// ****************************************************************************

void MainWindow::OnSetDevices(wxCommandEvent& event)
{
  int outputAudioDevice = lstAudioOutputDevice->GetCurrentSelection();
  int inputAudioDevice = lstAudioInputDevice->GetCurrentSelection();

  initSound(SAMPLING_RATE, inputAudioDevice, outputAudioDevice);
}


// ****************************************************************************
// ****************************************************************************

void MainWindow::OnGenerateSweep(wxCommandEvent& event)
{
  double freq_Hz[4] = { 0.0 };
  bool isValid[4];

  isValid[0] = txtFadeInLowerFreq->GetValue().ToDouble(&freq_Hz[0]);
  isValid[1] = txtFadeInUpperFreq->GetValue().ToDouble(&freq_Hz[1]);
  isValid[2] = txtFadeOutLowerFreq->GetValue().ToDouble(&freq_Hz[2]);
  isValid[3] = txtFadeOutUpperFreq->GetValue().ToDouble(&freq_Hz[3]);

  if ((!isValid[0]) || (!isValid[1]) || (!isValid[2]) || (!isValid[3]))
  {
    wxMessageBox("All frequency values must be valid numbers.", "Error");
    updateWidgets();
    return;
  }

  double amp = 0.0;
  if (txtSourceAmpFactor->GetValue().ToDouble(&amp) == false)
  {
    wxMessageBox("Invalid source amplitude!", "Error");
    updateWidgets();
    return;
  }
  else
  {
    data->sourceAmpFactor = amp;
  }

  // Safety checks.

  if (data->sourceAmpFactor < 0.0)
  {
    data->sourceAmpFactor = 0.0;
  }

  if (data->sourceAmpFactor > 1.0)
  {
    data->sourceAmpFactor = 1.0;
  }

  // Check that all 4 frequencies are in the allowed range.

  bool ok = true;
  for (int i = 0; i < 4; i++)
  {
    if (freq_Hz[i] < Data::SWEEP_FREQ_MIN)
    {
      ok = false;
    }

    if (freq_Hz[i] > Data::SWEEP_FREQ_MAX)
    {
      ok = false;
    }
  }

  if (ok == false)
  {
    wxString st = wxString::Format("All frequencies must be between %d and %d Hz!",
      Data::SWEEP_FREQ_MIN, Data::SWEEP_FREQ_MAX);
    wxMessageBox(st, "Error");
    updateWidgets();
    return;
  }

  // Check that the order of the frequencies is ok.

  if ((freq_Hz[0] >= freq_Hz[1]) || (freq_Hz[1] >= freq_Hz[2]) || (freq_Hz[2] >= freq_Hz[3]))
  {
    wxMessageBox("The frequency values are not in a valid order!", "Error");
    updateWidgets();
    return;
  }

  data->createExponentialSweep(freq_Hz);

  // ****************************************************************

  updateWidgets();
  picSignal->Refresh();
  picSpectrum->Refresh();
}


// ****************************************************************************
// ****************************************************************************

void MainWindow::OnPlaySourceSignal(wxCommandEvent& event)
{
  playSignal(data->sourceSignal);
}


// ****************************************************************************
// ****************************************************************************

void MainWindow::OnSignalSelection(wxCommandEvent& event)
{
  data->selectedResponse = (Data::MeasuredResponse)event.GetInt();
  updateWidgets();
 
  picSignal->Refresh();
  picImpulseResponse->Refresh();
}


// ****************************************************************************
// ****************************************************************************

void MainWindow::OnRecord(wxCommandEvent& event)
{
  Signal *s = data->sweepResponse[data->selectedResponse];

  data->playbackAndRecord(data->sourceSignal, s,
    (double)Data::SOURCE_SIGNAL_LENGTH_PT / (double)SAMPLING_RATE + 1.0);

  // Directly calculate the impulse response and spectrum.
  data->calcImpulseResponse();

  // ****************************************************************

  picSignal->Refresh();
  picImpulseResponse->Refresh();
  picSpectrum->Refresh();
  wxYield();

  // ****************************************************************
  // Show a warning if the max. signal amplitude of the recording
  // is below 5% or above 95% of the amplitude range.
  // ****************************************************************

  int i;
  double max = 0.0;
  for (i = 0; i < s->N; i++)
  {
    if (fabs(s->x[i]) > max)
    {
      max = fabs(s->x[i]);
    }
  }

  if (max < 0.05)
  {
    wxMessageBox("Warning! Max. signal amplitude is below 5% of amplitude range. "
      "Please adjust microphone or loudspeaker levels and repeat the measurement!",
      "Warning", wxICON_ERROR | wxOK);
  }
  else
  if (max > 0.95)
  {
    wxMessageBox("Warning! Max. signal amplitude is above 95% of amplitude range. "
      "Please adjust microphone or loudspeaker levels and repeat the measurement!",
      "Warning", wxICON_ERROR | wxOK);
  }
  else
  {
    wxMessageBox("You can now select the *linear* impulse response and determine its\n"
      "spectrum with the context menu in the impulse response display.", "Information!",
      wxICON_NONE | wxOK);
  }

}


// ****************************************************************************
// ****************************************************************************

void MainWindow::OnReplay(wxCommandEvent& event)
{
  playSignal(data->sweepResponse[data->selectedResponse]);
}


// ****************************************************************************
// Save a mono wav-file with the measured sweep response.
// ****************************************************************************

void MainWindow::OnSaveSignal(wxCommandEvent& event)
{
  wxFileDialog dialog(this, "Save wave file",
    audioFileName.GetPath(), audioFileName.GetFullName(),
    "Wave files (*.wav)|*.wav", wxFD_SAVE | wxFD_OVERWRITE_PROMPT);

  if (dialog.ShowModal() == wxID_OK)
  {
    audioFileName = wxFileName(dialog.GetPath());

    AudioFile<double> audioFile;
    const int NUM_CHANNELS = 1;
    int numSamples = data->sweepResponse[data->selectedResponse]->N;
    audioFile.setAudioBufferSize(NUM_CHANNELS, numSamples);

    for (int i = 0; i < numSamples; i++)
    {
      audioFile.samples[0][i] = data->sweepResponse[data->selectedResponse]->x[i];
    }

    audioFile.setSampleRate(SAMPLING_RATE);
    audioFile.setBitDepth(24);

    if (!audioFile.save(audioFileName.GetFullPath().ToStdString()))
    {
      wxMessageBox("Error saving the file.", "Error!");
    }
  }
}


// ****************************************************************************
/// Load a sweep response.
// ****************************************************************************

void MainWindow::OnLoadSignal(wxCommandEvent& event)
{
  wxFileDialog dialog(this, "Open wave file",
    audioFileName.GetPath(), audioFileName.GetFullName(),
    "Wave files (*.wav)|*.wav", wxFD_OPEN | wxFD_FILE_MUST_EXIST);

  if (dialog.ShowModal() == wxID_OK)
  {
    audioFileName = wxFileName(dialog.GetPath());

    AudioFile<double> audioFile;
    if (!audioFile.load(audioFileName.GetFullPath().ToStdString()))
    {
      wxMessageBox("Error in loading the file.", "Error!");
      return;
    }

    // **************************************************************
    // Check that the audio file has the right number of channels (1),
    // the right sampling rate and quantization.
    // **************************************************************

    wxString st;

    if (audioFile.getNumChannels() != 1)
    {
      st = wxString::Format("Wrong data format: The number of channels is %d but must be 1.",
        audioFile.getNumChannels());
      wxMessageBox(st, "Error!");
      return;
    }

    if (audioFile.getSampleRate() != SAMPLING_RATE)
    {
      st = wxString::Format("Wrong data format: The sampling rate is %d but must be %d Hz.",
        audioFile.getSampleRate(), SAMPLING_RATE);
      wxMessageBox(st, "Error!");
      return;
    }

    if (audioFile.getBitDepth() != 24)
    {
      st = wxString::Format("Wrong data format: The quantization is %d bit but must be 24 bit.",
        audioFile.getBitDepth());
      wxMessageBox(st, "Error!");
      return;
    }

    if (audioFile.getNumSamplesPerChannel() != data->sourceSignal->N)
    {
      st = wxString::Format("Wrong data format: The number of samples is %d but must be be %d.",
        audioFile.getNumSamplesPerChannel(), data->sourceSignal->N);
      wxMessageBox(st, "Error!");
      return;
    }

    // **************************************************************
    // Load the data as the selected sweep response.
    // **************************************************************

    int numSamples = data->sweepResponse[data->selectedResponse]->N;
    if (audioFile.getNumSamplesPerChannel() < numSamples)
    {
      numSamples = audioFile.getNumSamplesPerChannel();
    }

    data->sweepResponse[data->selectedResponse]->setZero();

    for (int i = 0; i < numSamples; i++)
    {
      data->sweepResponse[data->selectedResponse]->x[i] = audioFile.samples[0][i];
    }

    // **************************************************************
    // Automatically calculate the impulse response and spectrum and
    // tell the user to select the linear response.
    // **************************************************************

    data->calcImpulseResponse();

    picSignal->Refresh();
    picImpulseResponse->Refresh();
    picSpectrum->Refresh();
    
    wxYield();

    st = "The impulse response and the response spectrum have been calculated\n"
      "for the loaded sweep response based on the current source signal.\n"
      "Note that this is only correct when the source signal equals the one\n"
      "the loaded response has been recorded with!\n"
      "\n"
      "You can now select the *linear* impulse response and determine its\n"
      "spectrum with the context menu in the impulse response display.";

    wxMessageBox(st, "Information!");
  }

}


// ****************************************************************************
// ****************************************************************************

void MainWindow::OnSaveSpectrum(wxCommandEvent& event)
{
  wxString name = wxFileSelector("Save transfer function", exportFileName.GetPath(),
    exportFileName.GetFullName(), "txt", "Text files (*.txt)|*.txt",
    wxFD_SAVE | wxFD_OVERWRITE_PROMPT, this);

  if (name.empty() == false)
  {
    exportFileName = wxFileName(name);
    if (data->saveSpectrum(name, data->responseSpectrum[data->selectedResponse]) == false)
    {
      wxMessageBox("Error saving the file.", "Error!");
    }
  }
}


// ****************************************************************************
/// Calc. the quotient of the filtered spectrum and the direct spectrum.
// ****************************************************************************

void MainWindow::OnCalcFinalTransferFunction(wxCommandEvent& event)
{
  const double EPSILON = 0.000000001;
  int i;
  Complex y;
  double magnitude;
  double freq = 0.0;

  double normFactor = data->peakNormalizationFactor[Data::REFERENCE_RESPONSE] /
    data->peakNormalizationFactor[Data::PRIMARY_RESPONSE];

  wxPrintf("The magnitude is scaled by the factor %2.4f (= %2.4f / %2.4f) to undo "
    "the peak normalization of the impulse responses.\n",
    normFactor,
    data->peakNormalizationFactor[Data::REFERENCE_RESPONSE],
    data->peakNormalizationFactor[Data::PRIMARY_RESPONSE]);
  
  wxString message = "If you want to shift the level of the final transfer function,\n"
    "e.g., due to different sensitivities of the used microphones,\n"
    "please enter the level correction in dB (0.0 dB = no correction)";

  wxString text = wxGetTextFromUser(message, "Shift level in dB", "0.0", this);

  // If the user pressed "CANCEL", return.
  if (text.IsEmpty())
  {
    return;
  }

  double levelShift_dB = 0.0;
  if (text.ToDouble(&levelShift_dB) == false)
  {
    wxMessageBox("Invalid number. The level will not be shifted.", "Error");
    levelShift_dB = 0.0;
  }

  double levelShiftFactor = pow(10.0, levelShift_dB / 20.0);
  normFactor *= levelShiftFactor;

  // ****************************************************************
  // Init the final spectrum with zeros.
  // ****************************************************************

  int N = Data::CONVOLVED_SIGNAL_LENGTH_PT;
  data->finalTransferFunction->reset(N);

  for (i = 0; i < N; i++)
  {
    freq = (double)i * (double)SAMPLING_RATE / (double)N;

    // When we are outside the powerband of the sweep, set the value
    // of the final transfer function to 1.0. Here we must take
    // case that this is done for both the positive and negative 
    // frequencies!

    if ((freq < data->currSweepFreq_Hz[1]) || (freq > SAMPLING_RATE - data->currSweepFreq_Hz[1]) ||
      ((freq > data->currSweepFreq_Hz[2]) && (freq < SAMPLING_RATE - data->currSweepFreq_Hz[2])))
    {
      y = 1.0;
    }
    else

    // When we are in the powerband, calculate the quotient between 
    // both response spectra.
    {
      magnitude = data->responseSpectrum[Data::REFERENCE_RESPONSE]->getMagnitude(i);
      if (magnitude < EPSILON)
      {
        y = Complex(EPSILON, 0.0);
      }
      else
      {
        y = data->responseSpectrum[Data::REFERENCE_RESPONSE]->getValue(i);
      }

      y = data->responseSpectrum[Data::PRIMARY_RESPONSE]->getValue(i) / y;

      // Undo the scaling imposed by the peak-normalization of the 
      // impulse responses
      y *= normFactor;
    }

    // Set the value for the final transfer function.
    data->finalTransferFunction->setValue(i, y);
  }

  picSpectrum->Refresh();
}


// ****************************************************************************
// ****************************************************************************

void MainWindow::OnSaveFinalTransferFunction(wxCommandEvent& event)
{
  wxString name = wxFileSelector("Save transfer function", exportFileName.GetPath(),
    exportFileName.GetFullName(), "txt", "Text files (*.txt)|*.txt",
    wxFD_SAVE | wxFD_OVERWRITE_PROMPT, this);

  if (name.empty() == false)
  {
    exportFileName = wxFileName(name);
    if (data->saveSpectrum(name, data->finalTransferFunction) == false)
    {
      wxMessageBox("Error saving the file.", "Error!");
    }
  }
}


// ****************************************************************************
// ****************************************************************************

void MainWindow::OnScrollTime1(wxScrollEvent& event)
{
  int pos = event.GetPosition();
  double pos_s = (double)pos / 100.0;

  picSignal->graph.abscissa.reference = pos_s;

  updateWidgets();
  picSignal->Refresh();
}


// ****************************************************************************
// ****************************************************************************

void MainWindow::OnScrollTime2(wxScrollEvent& event)
{
  int pos = event.GetPosition();
  double pos_s = (double)pos / 100.0;

  picImpulseResponse->graph.abscissa.reference = pos_s;

  updateWidgets();
  picImpulseResponse->Refresh();
}


// ****************************************************************************

