// ****************************************************************************
// Copyright(C) 2019 by Peter Birkholz, Dresden, Germany
// This file is part of the program MeasureTransferFunction.
// www.vocaltractlab.de
// ****************************************************************************

#include "ImpulseResponsePicture.h"

const wxColor ImpulseResponsePicture::IMPULSE_RESPONSE_COLOR[Data::NUM_RESPONSE_SIGNALS] = 
{ *wxRED, *wxBLUE };

// ****************************************************************************
// IDs.
// ****************************************************************************

static const int IDM_ZOOM_IN = 1000;
static const int IDM_ZOOM_OUT = 1001;
static const int IDM_SHOW_LOG_WAVEFORM = 1002;

static const int IDM_CLEAR_SELECTION = 1010;
static const int IDM_SET_SELECTION_START  = 1011;
static const int IDM_SET_SELECTION_END    = 1012;
static const int IDM_SELECT_ALL           = 1013;

static const int IDM_GO_TO_IMPULSE_RESPONSE = 1020;
static const int IDM_SELECTION_TO_SPECTRUM = 1021;

// ****************************************************************************
// The event table.
// ****************************************************************************

BEGIN_EVENT_TABLE(ImpulseResponsePicture, BasicPicture)
  EVT_MOUSE_EVENTS(OnMouseEvent)

  EVT_MENU(IDM_ZOOM_IN, OnZoomIn)
  EVT_MENU(IDM_ZOOM_OUT, OnZoomOut)
  EVT_MENU(IDM_SHOW_LOG_WAVEFORM, OnShowLogWaveform)

  EVT_MENU(IDM_CLEAR_SELECTION, OnClearSelection)
  EVT_MENU(IDM_SET_SELECTION_START, OnSetSelectionStart)
  EVT_MENU(IDM_SET_SELECTION_END, OnSetSelectionEnd)
  EVT_MENU(IDM_SELECT_ALL, OnSelectAll)
  
  EVT_MENU(IDM_GO_TO_IMPULSE_RESPONSE, OnGoToImpulseResponse)
  EVT_MENU(IDM_SELECTION_TO_SPECTRUM, OnSelectionToSpectrum)
  END_EVENT_TABLE()


// ****************************************************************************
/// Construcor. Passes the parent parameter.
// ****************************************************************************

ImpulseResponsePicture::ImpulseResponsePicture(wxWindow *parent) : BasicPicture(parent)
{
  // The time axis.

  graph.init(this, 40, 0, 0, 25);

  graph.initAbscissa(PQ_TIME, 0.0, 0.001,
    0.0, 0.0, 0.0, 0.01, Data::TRACK_LENGTH_S, 2.0,
    10, 3, false, false, true);

  graph.initLinearOrdinate(PQ_NONE, 0.0, 0.1,
    -1.0, -1.0, -1.0, 1.0, 1.0, 1.0,
    1, 1, false, false, true);

  // Set the viewport such that the linear impulse response is in the
  // viewport.

  double viewLength_s = 2.0;
  double time_s = Data::SOURCE_SIGNAL_LENGTH_PT / (double)SAMPLING_RATE - 0.5 * viewLength_s;
  graph.abscissa.reference = time_s;
  graph.abscissa.positiveLimit = viewLength_s;

  // Init some variables.
  
  data = Data::getInstance();
  menuX = 0;
  menuY = 0;
  lastMx = 0;
  lastMy = 0;
  moveLeftBorder = false;
  moveRightBorder = false;
  showLogWaveform = true;

  // Init the context menu.
  
  contextMenu = new wxMenu();
  contextMenu->Append(IDM_ZOOM_IN, "Zoom in");
  contextMenu->Append(IDM_ZOOM_OUT, "Zoom out");
  contextMenu->AppendCheckItem(IDM_SHOW_LOG_WAVEFORM, "Show log. waveform (dB)");
  contextMenu->AppendSeparator();

  contextMenu->Append(IDM_CLEAR_SELECTION, "Clear selection");
  contextMenu->Append(IDM_SET_SELECTION_START, "Set selection start");
  contextMenu->Append(IDM_SET_SELECTION_END, "Set selection end");
  contextMenu->Append(IDM_SELECT_ALL, "Select all");
  contextMenu->AppendSeparator();

  contextMenu->Append(IDM_GO_TO_IMPULSE_RESPONSE, "Go to impulse response");
  contextMenu->Append(IDM_SELECTION_TO_SPECTRUM, "Selection to spectrum");
}


// ****************************************************************************
// ****************************************************************************

void ImpulseResponsePicture::draw(wxDC &dc)
{
  paintOscillograms(dc);
}


// ****************************************************************************
/// Draws a mark at the left or right margin of the selected region.
// ****************************************************************************

void ImpulseResponsePicture::drawSelectionMark(wxDC &dc, int x, int y1, int y2, bool isLeftMark)
{
  wxPen oldPen = dc.GetPen();
  
  dc.SetPen(wxPen(*wxBLACK, 1, wxPENSTYLE_DOT));
  dc.DrawLine(x, y1, x, y2);

  dc.SetPen(*wxBLACK_PEN);
  dc.SetBrush(*wxBLACK_BRUSH);
  const int W = 8;
  
  if (isLeftMark)
  {
    wxPoint topLeftTriangle[3] = { wxPoint(x, y1), wxPoint(x+W, y1), wxPoint(x, y1+W) };
    wxPoint bottomLeftTriangle[3] = { wxPoint(x, y2), wxPoint(x+W, y2), wxPoint(x, y2-W) };
    dc.DrawPolygon(3, topLeftTriangle);
    dc.DrawPolygon(3, bottomLeftTriangle);
  }
  else
  {
    wxPoint topRightTriangle[3] = { wxPoint(x, y1), wxPoint(x-W, y1), wxPoint(x, y1+W) };
    wxPoint bottomRightTriangle[3] = { wxPoint(x, y2), wxPoint(x-W, y2), wxPoint(x, y2-W) };
    dc.DrawPolygon(3, topRightTriangle);
    dc.DrawPolygon(3, bottomRightTriangle);
  }

  dc.SetPen(oldPen);
}

// ****************************************************************************
// ****************************************************************************

void ImpulseResponsePicture::paintOscillograms(wxDC &dc)
{
  Data *data = Data::getInstance();

  int x;
  int graphX, graphY, graphW, graphH;
  int width, height;

  graph.getDimensions(graphX, graphY, graphW, graphH);
  this->GetSize(&width, &height);


  // ****************************************************************
  // Clear the background and paint the axes.
  // ****************************************************************

  dc.SetBackground(*wxWHITE_BRUSH);
  dc.Clear();
  graph.paintAbscissa(dc);
  graph.paintOrdinate(dc);


  // ****************************************************************
  // Overly the selected region.
  // ****************************************************************
  
  if (data->isValidSelection())
  {
    int x1, x2;

    dc.SetPen(wxPen(*wxBLACK, 1, wxPENSTYLE_TRANSPARENT));    // no pen
    dc.SetBrush(wxBrush(wxColor(220, 220, 220)));

    // Draw the upper and lower horizontal bars in the selected region.
    
    x1 = graph.getXPos(data->selectionMark_s[0]);
    x2 = graph.getXPos(data->selectionMark_s[1]);

    if ((x1 < graphX + graphW) && (x2 >= graphX))
    {
      if (x1 < graphX)
      {
        x1 = graphX;
      }
      if (x2 >= graphX + graphW)
      {
        x2 = graphX + graphW - 1;
      }

      dc.DrawRectangle(x1, graphY, x2 - x1, graphH/8);
      dc.DrawRectangle(x1, graphY + 7*graphH/8, x2 - x1, graphH/8);
    }

    // Draw the left taper region gray.
    
    x1 = graph.getXPos(data->selectionMark_s[0]);
    if (data->selectionMark_s[1] > data->selectionMark_s[0] + Data::TAPER_LENGTH_S)
    {
      x2 = graph.getXPos(data->selectionMark_s[0] + Data::TAPER_LENGTH_S);
    }
    else
    {
      x2 = graph.getXPos(data->selectionMark_s[1]);
    }

    if ((x1 < graphX + graphW) && (x2 >= graphX))
    {
      if (x1 < graphX)
      {
        x1 = graphX;
      }
      if (x2 >= graphX + graphW)
      {
        x2 = graphX + graphW - 1;
      }

      dc.DrawRectangle(x1, graphY, x2 - x1, graphH);
    }

    // Draw the right taper region gray.

    x2 = graph.getXPos(data->selectionMark_s[1]);
    if (data->selectionMark_s[0] < data->selectionMark_s[1] - Data::TAPER_LENGTH_S)
    {
      x1 = graph.getXPos(data->selectionMark_s[1] - Data::TAPER_LENGTH_S);
    }
    else
    {
      x1 = graph.getXPos(data->selectionMark_s[0]);
    }

    if ((x1 < graphX + graphW) && (x2 >= graphX))
    {
      if (x1 < graphX)
      {
        x1 = graphX;
      }
      if (x2 >= graphX + graphW)
      {
        x2 = graphX + graphW - 1;
      }

      dc.DrawRectangle(x1, graphY, x2 - x1, graphH);
    }
  }

  // ****************************************************************
  // Draw the selected impulse response.
  // ****************************************************************

  paintOscillogram(dc, data->impulseResponse[data->selectedResponse], IMPULSE_RESPONSE_COLOR[data->selectedResponse]);

  // ****************************************************************
  // Paint the two selection marks and the main mark.
  // ****************************************************************

  x = graph.getXPos(data->selectionMark_s[0]);
  if ((data->selectionMark_s[0] >= 0) && (x >= graphX) && (x < graphX + graphW))
  {
    drawSelectionMark(dc, x, graphY, graphY + graphH - 1, true);
  }

  x = graph.getXPos(data->selectionMark_s[1]);
  if ((data->selectionMark_s[1] >= 0) && (x >= graphX) && (x < graphX + graphW))
  {
    drawSelectionMark(dc, x, graphY, graphY + graphH - 1, false);
  }

  // ****************************************************************
  // Output some text.
  // ****************************************************************

  dc.SetPen(*wxBLACK_PEN);
  dc.SetBackgroundMode(wxSOLID);    // Set a solid white background
  dc.SetTextBackground(wxColor(240, 240, 240));
  wxString st;

  st = wxString::Format(
    "View range: %2.3f s",
    graph.abscissa.positiveLimit);
  dc.DrawText(st, graphX + 3, 0);

  dc.SetTextForeground(IMPULSE_RESPONSE_COLOR[data->selectedResponse]);
  if (data->selectedResponse == Data::PRIMARY_RESPONSE)
  {
    dc.DrawText("Primary impulse response", graphX + 3, 20);
  }
  else
  {
    dc.DrawText("Reference impulse response", graphX + 3, 20);
  }


  dc.SetPen(*wxBLACK_PEN);

  if (data->isValidSelection())
  {
    double left = data->selectionMark_s[0];
    double right = data->selectionMark_s[1];
    double denominator = right - left;
    if (denominator == 0.0) 
    { 
      denominator = 0.000001; 
    }
    double f = 1.0 / denominator;

    st = wxString::Format("Selection: %2.3f ... %2.3f s (%2.3f s = %2.1f Hz)",
      left,
      right,
      right-left,
      f);
    dc.DrawText(st, 320, 0);
  }

}


// ****************************************************************************
// Draw a single oscillogram.
// ****************************************************************************

void ImpulseResponsePicture::paintOscillogram(wxDC &dc, Signal *s, const wxColor &color)
{
  int i, k;
  int graphX, graphY, graphW, graphH;
  double tLeft_s, tRight_s;
  int sampleIndexLeft, sampleIndexRight;
  double lastSampleValue = 0.0;
  double value;
  double minValue, maxValue;
  int minY, maxY;

  graph.getDimensions(graphX, graphY, graphW, graphH);

  dc.SetPen(wxPen(color));

  // ****************************************************************
  // Run through all pixels from left to right.
  // ****************************************************************

  for (i = 0; i < graphW; i++)
  {
    // Time and audio sampling index at the left and right edge of a pixel.
    tLeft_s = graph.getAbsXValue(graphX + i);
    tRight_s = graph.getAbsXValue(graphX + i + 1);
    sampleIndexLeft = (int)(tLeft_s * SAMPLING_RATE);
    sampleIndexRight = (int)(tRight_s * SAMPLING_RATE);

    if ((sampleIndexLeft >= 0) && (sampleIndexRight < s->N))
    {
      minValue = lastSampleValue;
      maxValue = lastSampleValue;

      for (k = sampleIndexLeft; k <= sampleIndexRight; k++)
      {
        value = getSample(s, k);
        if (value < minValue)
        {
          minValue = value;
        }
        if (value > maxValue)
        {
          maxValue = value;
        }
      }

      lastSampleValue = getSample(s, sampleIndexRight);

      minY = graph.getYPos(maxValue);
      maxY = graph.getYPos(minValue);

      if (minY < graphY)
      {
        minY = graphY;
      }
      if (minY >= graphY + graphH)
      {
        minY = graphY + graphH - 1;
      }

      if (maxY < graphY)
      {
        maxY = graphY;
      }
      if (maxY >= graphY + graphH)
      {
        maxY = graphY + graphH - 1;
      }

      if (minY == maxY)
      {
        dc.DrawPoint(graphX + i, minY);
      }
      else
      {
        dc.DrawLine(graphX + i, minY, graphX + i, maxY);
      }
    }
  }
}


// ****************************************************************************
/// Returns the sample at the given position index in the given signal s either
/// as it is (linear scale) or as log. value (when showLogWaveform = true).
// ****************************************************************************

inline double ImpulseResponsePicture::getSample(Signal *s, int index)
{
  const double EPSILON = 0.000001;
  const double REFERENCE = 0.0005;
  const double FACTOR = 1.0 / log(1.0 / REFERENCE);

  double d = s->x[index];

  if (showLogWaveform)
  {
    if (d > REFERENCE + EPSILON)
    {
      d = FACTOR * log(d / REFERENCE);
    }
    else
    if (d < -REFERENCE - EPSILON)
    {
      d = -FACTOR * log(-d / REFERENCE);
    }
    else
    {
      d = 0.0;
    }
  }
  
  return d;
}


// ****************************************************************************
/// Process all mouse events.
// ****************************************************************************

void ImpulseResponsePicture::OnMouseEvent(wxMouseEvent &event)
{
  int width, height;
  this->GetSize(&width, &height);

  int mx = event.GetX();
  int my = event.GetY();

  // ****************************************************************
  // The x-position in pixels of the selection region borders.
  // ****************************************************************

  int leftBorderX = graph.getXPos(data->selectionMark_s[0]);
  int rightBorderX = graph.getXPos(data->selectionMark_s[1]);
  bool isOnLeftBorder = false;
  bool isOnRightBorder = false;

  int M = 4;
  if ((mx > leftBorderX - M) && (mx < leftBorderX + M))
  {
    isOnLeftBorder = true;
  }
  else
  if ((mx > rightBorderX - M) && (mx < rightBorderX + M))
  {
    isOnRightBorder = true;
  }

  // ****************************************************************
  // The mouse is entering the window.
  // ****************************************************************

  if ((event.Entering()) || (event.Leaving()))
  {
    moveLeftBorder = false;
    moveRightBorder = false;
    lastMx = mx;
    lastMy = my;
    return;
  }

  // ****************************************************************
  // This a button-up event.
  // ****************************************************************

  if (event.ButtonUp())
  {
    moveLeftBorder = false;
    moveRightBorder = false;
    lastMx = mx;
    lastMy = my;
    return;
  }

  // ****************************************************************
  // The left mouse button changed to down.
  // ****************************************************************

  if (event.LeftDown())
  {
    if (isOnLeftBorder)
    {
      moveLeftBorder = true;
    }
    else
    if (isOnRightBorder)
    {
      moveRightBorder = true;
    }
    else
    {
      // ...
    }

    lastMx = mx;
    lastMy = my;
    return;
  }

  // ****************************************************************
  // Was the mouse moved? The possibly change the cursor type.   
  // ****************************************************************

  if (event.Moving())
  {
    // Change the type of the cursor when a border of the selection 
    // region is under the cursor.
    if ((isOnLeftBorder) || (isOnRightBorder) || (moveLeftBorder) || (moveRightBorder))
    {
      this->SetCursor( wxCursor(wxCURSOR_SIZEWE) );
    }
    else
    {
      this->SetCursor( wxCursor(wxCURSOR_ARROW) );
    }
  }

  // ****************************************************************
  // The mouse is dragged (with one or more mouse buttons pressed).
  // ****************************************************************

  if (event.Dragging())
  {
    if (moveLeftBorder)
    {
      data->selectionMark_s[0] = graph.getAbsXValue(mx);

      if (data->selectionMark_s[0] < 0.0)
      {
        data->selectionMark_s[0] = 0.0;
      }
      if (data->selectionMark_s[0] > Data::TRACK_LENGTH_S)
      {
        data->selectionMark_s[0] = Data::TRACK_LENGTH_S;
      }

      this->Refresh();
    }
    else
    if (moveRightBorder)
    {
      data->selectionMark_s[1] = graph.getAbsXValue(mx);

      if (data->selectionMark_s[1] < 0.0)
      {
        data->selectionMark_s[1] = 0.0;
      }
      if (data->selectionMark_s[1] > Data::TRACK_LENGTH_S)
      {
        data->selectionMark_s[1] = Data::TRACK_LENGTH_S;
      }

      this->Refresh();
    }
    else
    {
      // ...

    }
  }

  // ****************************************************************
  // The right mouse button changed to down. Call the context menu.
  // ****************************************************************

  if (event.RightDown())
  {
    menuX = mx;
    menuY = my;
    
    contextMenu->Check(IDM_SHOW_LOG_WAVEFORM, showLogWaveform);

    PopupMenu(contextMenu);
    return;
  }

}


// ****************************************************************************
// ****************************************************************************

void ImpulseResponsePicture::OnZoomIn(wxCommandEvent &event)
{
  graph.zoomInAbscissa(false, true);
  this->Refresh();
}


// ****************************************************************************
// ****************************************************************************

void ImpulseResponsePicture::OnZoomOut(wxCommandEvent &event)
{
  graph.zoomOutAbscissa(false, true);
  this->Refresh();
}


// ****************************************************************************
// ****************************************************************************

void ImpulseResponsePicture::OnShowLogWaveform(wxCommandEvent &event)
{
  showLogWaveform = !showLogWaveform;
  this->Refresh();
}


// ****************************************************************************
// ****************************************************************************

void ImpulseResponsePicture::OnClearSelection(wxCommandEvent &event)
{
  data->selectionMark_s[0] = -1.0;
  data->selectionMark_s[1] = -1.0;
  this->Refresh();
}


// ****************************************************************************
// ****************************************************************************

void ImpulseResponsePicture::OnSetSelectionStart(wxCommandEvent &event)
{
  data->selectionMark_s[0] = graph.getAbsXValue(menuX);
  if (data->selectionMark_s[0] < 0.0)
  {
    data->selectionMark_s[0] = 0.0;
  }
  this->Refresh();
}


// ****************************************************************************
// ****************************************************************************

void ImpulseResponsePicture::OnSetSelectionEnd(wxCommandEvent &event)
{
  data->selectionMark_s[1] = graph.getAbsXValue(menuX);
  if (data->selectionMark_s[1] > Data::TRACK_LENGTH_S)
  {
    data->selectionMark_s[1] = Data::TRACK_LENGTH_S;
  }
  this->Refresh();
}

// ****************************************************************************
// ****************************************************************************

void ImpulseResponsePicture::OnSelectAll(wxCommandEvent &event)
{
  data->selectionMark_s[0] = 0.0;
  data->selectionMark_s[1] = Data::TRACK_LENGTH_S;
  this->Refresh();
}


// ****************************************************************************
// ****************************************************************************

void ImpulseResponsePicture::OnGoToImpulseResponse(wxCommandEvent &event)
{
  double viewLength_s = 2.0;
  double time_s = Data::SOURCE_SIGNAL_LENGTH_PT / (double)SAMPLING_RATE - 0.5 * viewLength_s;

  graph.abscissa.reference = time_s;
  graph.abscissa.positiveLimit = viewLength_s;

  // Also update the time scroll bar in the parent window.
  wxCommandEvent newEvent(updateRequestEvent);
  event.SetInt(UPDATE_WIDGETS);
  wxPostEvent(this->GetParent(), newEvent);

  this->Refresh();
}


// ****************************************************************************
// ****************************************************************************

void ImpulseResponsePicture::OnSelectionToSpectrum(wxCommandEvent &event)
{
    if (data->selectionMark_s[1] - data->selectionMark_s[0] < 2.0 * Data::TAPER_LENGTH_S)
    {
      wxString st = wxString::Format("The time window must be at least %2.1f ms long.",
        (2.0 * Data::TAPER_LENGTH_S)*1000.0);
      wxMessageBox(st, "Error");
      return;
    }

    data->calcSmoothedSpectrum();

    // Refresh all pictures on the main window.
    this->GetParent()->Refresh();
}

// ****************************************************************************

