// ****************************************************************************
// Copyright(C) 2019 by Peter Birkholz, Dresden, Germany
// This file is part of the program MeasureTransferFunction.
// www.vocaltractlab.de
// ****************************************************************************

#ifndef ___DATA__
#define ___DATA__

#include <wx/wx.h>
#include <string>
#include <vector>
#include "Signal.h"
#include "Dsp.h"

// ****************************************************************************
// Declare some custom event types.
// The updateRequestEvent is meant to be used by child widgets to request their
// parent to update some region or other child widgets.
// A command event of this type is posted as follows:
//
//   wxCommandEvent event(updateRequestEvent);
//   event.SetInt(REFRESH_PICTURES | REFRESH_PICTURES_AND_CONTROLS);
//   wxPostEvent(receiverWindow, event);
//
// The receiver window must have the following in his event table:
//
//  EVT_COMMAND(wxID_ANY, updateRequestEvent, OnUpdateRequest)
// 
// The function OnUpdateRequest(...) takes one wxCommandEvent parameter.
// ****************************************************************************

extern const wxEventType updateRequestEvent;
extern const int UPDATE_WIDGETS;

using namespace std;


// ****************************************************************************
/// Singleton class containing the data and common methods for the frontend 
/// classes.
// ****************************************************************************

class Data
{
  // **************************************************************************
  // Public data.
  // **************************************************************************

public:
  enum MeasuredResponse
  {
    REFERENCE_RESPONSE = 0,
    PRIMARY_RESPONSE,
    NUM_RESPONSE_SIGNALS
  };

  static const int SWEEP_FREQ_MIN = 20;
  static const int SWEEP_FREQ_MAX = 24000;
  static const int DEFAULT_FADE_IN_LOWER_FREQ = 50;
  static const int DEFAULT_FADE_IN_UPPER_FREQ = 100;
  static const int DEFAULT_FADE_OUT_LOWER_FREQ = 20000;
  static const int DEFAULT_FADE_OUT_UPPER_FREQ = 21000;

  static const int TRACK_LENGTH_S = 60;
  static const int SOURCE_SIGNAL_LENGTH_EXPONENT = 20;
  static const int SOURCE_SIGNAL_LENGTH_PT = (int)1 << SOURCE_SIGNAL_LENGTH_EXPONENT;
  static const int CONVOLVED_SIGNAL_LENGTH_EXPONENT = 21;
  static const int CONVOLVED_SIGNAL_LENGTH_PT = (int)1 << CONVOLVED_SIGNAL_LENGTH_EXPONENT;

  static const double TAPER_LENGTH_S;

  double selectionMark_s[2];
  double mark_s;
  MeasuredResponse selectedResponse;
  double sourceAmpFactor;
  double currSweepFreq_Hz[4];   // Two for fade-in and two for fade-out.

  Signal *sourceSignal;
  Signal *inverseSourceSignal;
  Signal *sweepResponse[NUM_RESPONSE_SIGNALS];
  Signal *impulseResponse[NUM_RESPONSE_SIGNALS];
  double peakNormalizationFactor[NUM_RESPONSE_SIGNALS];

  ComplexSignal *sourceSpectrum;
  ComplexSignal *inverseSourceSpectrum;
  ComplexSignal *sourceProductSpectrum;
  ComplexSignal *responseSpectrum[NUM_RESPONSE_SIGNALS];
  ComplexSignal *finalTransferFunction;

  /// Path to the executable file
  wxString programPath;
  
  // **************************************************************************
  // Public functions.
  // **************************************************************************

public:
  static Data *getInstance();
  void init(const wxString &arg0);
  bool isValidSelection();

  void createExponentialSweep(double freq_Hz[]);

  void playbackAndRecord(Signal *source, Signal *target, double duration_s);
  void calcSpectrum(Signal *s, ComplexSignal *spectrum);
  void calcImpulseResponse();
  void calcSmoothedSpectrum();

  bool saveSpectrum(const wxString &fileName, ComplexSignal *spectrum);

  // **************************************************************************
  // Private data.       
  // **************************************************************************

private:
  static Data *instance;
  ComplexSignal *tempSpectrum;

  // **************************************************************************
  // Private functions.
  // **************************************************************************

private:
  Data();
};

#endif

// ****************************************************************************

