// ****************************************************************************
// Copyright(C) 2019 by Peter Birkholz, Dresden, Germany
// This file is part of the program MeasureTransferFunction.
// www.vocaltractlab.de
// ****************************************************************************

#include <wx/filename.h>
#include <wx/stdpaths.h>
#include <wx/progdlg.h>
#include <iostream>
#include <fstream>
#include <wx/busyinfo.h>

#include "Data.h"
#include "SoundLib.h"

// Define a custom event type to be used for command events.

const wxEventType updateRequestEvent = wxNewEventType();
const int UPDATE_WIDGETS = 0;

// ****************************************************************************
// Static data.
// ****************************************************************************

Data *Data::instance = NULL;

// 10 ms for tapering off the signal before the FFT.
const double Data::TAPER_LENGTH_S = 0.01;   

// ****************************************************************************
/// Returns the one instance of this class.
// ****************************************************************************

Data *Data::getInstance()
{
  if (instance == NULL)
  {
    instance = new Data();
  }
  return instance;
}


// ****************************************************************************
/// Init the data. This function must be called once after the first call of
/// getInstance().
/// \param arg0 The first string parameter passed to this program.
// ****************************************************************************

void Data::init(const wxString &arg0)
{
  int i;

  // ****************************************************************
  // Determine the program path from arg0. The option
  // wxPATH_GET_SEPARATOR makes sure that the path is always
  // terminated with a "\".
  // ****************************************************************

  wxFileName fileName(arg0);
  programPath = fileName.GetPath(wxPATH_GET_VOLUME | wxPATH_GET_SEPARATOR);
  printf("The program path is %s.\n", (const char*)programPath.mb_str());

  // ****************************************************************
  // Init variables.
  // ****************************************************************

  selectionMark_s[0] = -1.0;
  selectionMark_s[1] = -1.0;
  mark_s = 0.0;
  selectedResponse = REFERENCE_RESPONSE;
  sourceAmpFactor = 0.5;
  
  currSweepFreq_Hz[0] = Data::DEFAULT_FADE_IN_LOWER_FREQ;     // Start of fade-in
  currSweepFreq_Hz[1] = Data::DEFAULT_FADE_IN_UPPER_FREQ;     // End of fade-in
  currSweepFreq_Hz[2] = Data::DEFAULT_FADE_OUT_LOWER_FREQ;    // Start of fade-out
  currSweepFreq_Hz[3] = Data::DEFAULT_FADE_OUT_UPPER_FREQ;    // End of fade-out

  sourceSignal = new Signal(SOURCE_SIGNAL_LENGTH_PT);
  inverseSourceSignal = new Signal(SOURCE_SIGNAL_LENGTH_PT);

  sourceSpectrum = new ComplexSignal(CONVOLVED_SIGNAL_LENGTH_PT);
  inverseSourceSpectrum = new ComplexSignal(CONVOLVED_SIGNAL_LENGTH_PT);
  sourceProductSpectrum = new ComplexSignal(CONVOLVED_SIGNAL_LENGTH_PT);
  finalTransferFunction = new ComplexSignal(CONVOLVED_SIGNAL_LENGTH_PT);

  // All spectra are set to 1.0.
  sourceSpectrum->setToOne();
  inverseSourceSpectrum->setToOne();
  sourceProductSpectrum->setToOne();
  finalTransferFunction->setToOne();

  for (i = 0; i < NUM_RESPONSE_SIGNALS; i++)
  {
    sweepResponse[i] = new Signal(SOURCE_SIGNAL_LENGTH_PT);
    impulseResponse[i] = new Signal(CONVOLVED_SIGNAL_LENGTH_PT);
    responseSpectrum[i] = new ComplexSignal(CONVOLVED_SIGNAL_LENGTH_PT);
    responseSpectrum[i]->setToOne();
    peakNormalizationFactor[i] = 1.0;
  }

  tempSpectrum = new ComplexSignal(CONVOLVED_SIGNAL_LENGTH_PT);

  // Generate the default source sweep.
  createExponentialSweep(currSweepFreq_Hz);
}


// ****************************************************************************
// ****************************************************************************

bool Data::isValidSelection()
{
  if ((selectionMark_s[0] < selectionMark_s[1]) &&
    (selectionMark_s[0] >= 0.0) &&
    (selectionMark_s[1] >= 0.0))
  {
    return true;
  }
  else
  {
    return false;
  }
}


// ****************************************************************************
/// Create an exponential sweep between freq_Hz[0] and freq_Hz[3], the
/// inverse sweep, and the corresponding spectra.
/// There is a fading-in between freq_Hz[0] and freq_Hz[1] and
/// a fading-out between freq_Hz[2] and freq_Hz[3].
/// The method is based on the paper by Delvaux and Howard (2014), PLOS ONE,
/// and on my lecture slides on sweeps for Signal Processing.
// ****************************************************************************

void Data::createExponentialSweep(double freq_Hz[])
{
  int i;
  double y;
  double phi;

  // ********************************************************************
  // The new given frequencies are the new current frequencies.
  // ********************************************************************

  for (i = 0; i < 4; i++)
  {
    currSweepFreq_Hz[i] = freq_Hz[i];
  }

  // ********************************************************************
  // The sweep length is the source signal length minus a short
  // silence period at the beginning and the end.
  // ********************************************************************

  double silenceDuration_s = 0.5;
  int numSilenceSamples = (int)(silenceDuration_s * (double)SAMPLING_RATE);
  int firstSweepSample = numSilenceSamples;
  int numSweepSamples = sourceSignal->N - 2 * numSilenceSamples;

  double normFreq[4];
  double timeStep_s = 1.0 / (double)SAMPLING_RATE;

  for (i = 0; i < 4; i++)
  {
    normFreq[i] = 2.0 * M_PI * freq_Hz[i] * timeStep_s;
  }

  double sweepRate = log(normFreq[3] / normFreq[0]);

  // Generate the sweep samples.

  sourceSignal->setZero();

  for (i = 0; i < numSweepSamples; i++)
  {
    phi = (normFreq[0] * numSweepSamples / sweepRate) * (exp(i*sweepRate / numSweepSamples) - 1.0);
    y = sourceAmpFactor * sin(phi);
    sourceSignal->setValue(firstSweepSample + i, y);
  }


  // ********************************************************************
  // Create a "soft on" and "soft off" of the signal with a raised cosine.
  // ********************************************************************

  int numFadeInSamples = (int)(numSweepSamples * log(normFreq[1] / normFreq[0]) / log(normFreq[3] / normFreq[0]));
  int numFadeOutSamples = numSweepSamples - (int)(numSweepSamples * log(normFreq[2] / normFreq[0]) / log(normFreq[3] / normFreq[0]));
  int index;

  double w = 0.0;

  for (i = 0; i < numFadeInSamples; i++)
  {
    w = 0.5 * (1.0 - cos(i*M_PI / numFadeInSamples));
    index = firstSweepSample + i;
    y = sourceSignal->getValue(index);
    sourceSignal->setValue(index, y * w);
  }

  for (i = 0; i < numFadeOutSamples; i++)
  {
    w = 0.5 * (1.0 + cos(i*M_PI / numFadeOutSamples));
    index = firstSweepSample + numSweepSamples - numFadeOutSamples + i;
    y = sourceSignal->getValue(index);
    sourceSignal->setValue(index, y * w);
  }


  // ********************************************************************
  // Create the inverse filter (flip signal from left to right and apply
  // a decay of -6 dB/oct).
  // ********************************************************************

  double ampFactor = 0.0;

  inverseSourceSignal->setZero();

  for (i = 0; i < numSweepSamples; i++)
  {
    y = sourceSignal->getValue(firstSweepSample + numSweepSamples - 1 - i);
    ampFactor = 1.0 / (exp((double)i * sweepRate / numSweepSamples));
    inverseSourceSignal->setValue(firstSweepSample + i, y * ampFactor);
  }

  // ********************************************************************
  // Calculate the spectra of the sweep and the inverse sweep.
  // ********************************************************************

  calcSpectrum(sourceSignal, sourceSpectrum);
  calcSpectrum(inverseSourceSignal, inverseSourceSpectrum);

  *sourceProductSpectrum = *sourceSpectrum;
  *sourceProductSpectrum *= *inverseSourceSpectrum;
}


// ****************************************************************************
// ****************************************************************************

void Data::playbackAndRecord(Signal *source, Signal *target, double duration_s)
{
  int i;
  TemplateSignal<float> sourceFloat(source->N);
  TemplateSignal<float> targetFloat(target->N);

  // ****************************************************************
  // Convert the source signal from double to float.
  // ****************************************************************

  for (i = 0; i < source->N; i++)
  {
    sourceFloat.x[i] = (float)source->x[i];
  }

  // ****************************************************************
  // Start recording and playback.
  // ****************************************************************

  if (waveStartRecording(targetFloat.x, targetFloat.N) == false)
  {
    wxMessageBox("Failed to start recording!", "Error");
    return;
  }

  if (waveStartPlaying(sourceFloat.x, sourceFloat.N) == false)
  {
    wxMessageBox("Failed to start playing!", "Error");
    return;
  }

  // ****************************************************************
  // Show the busy dialog, wait for the specified duration and stop.
  // ****************************************************************

  wxWindowDisabler disableAll;
  wxBusyInfo wait("Please wait until the measurement is finished.");
  wxMilliSleep(100);
  wxTheApp->Yield();

  // Wait for the specified duration.

  wxMilliSleep((unsigned long)(duration_s*1000.0));

  waveStopRecording();
  waveStopPlaying();

  // ****************************************************************
  // Convert the target signal from float to double.
  // ****************************************************************

  for (i = 0; i < source->N; i++)
  {
    target->x[i] = (double)targetFloat.x[i];
  }

}


// ****************************************************************************
/// Calculates the FFT for the signal s with the length SOURCE_SIGNAL_LENGTH_PT.
// ****************************************************************************

void Data::calcSpectrum(Signal *s, ComplexSignal *spectrum)
{
  // The length of the spectrum must be that of the final convolved signal.
  spectrum->reset(CONVOLVED_SIGNAL_LENGTH_PT);

  // Put the time signal into the real part of the spectrum
  int i;
  for (i = 0; i < SOURCE_SIGNAL_LENGTH_PT; i++)
  {
    spectrum->re[i] = s->x[i];
  }

  // Calculate the actual FFT.
  realFFT(*spectrum, CONVOLVED_SIGNAL_LENGTH_EXPONENT, false);
}


// ****************************************************************************
/// Calculate the impulse response and the response spectrum for the selected
/// response signal.
// ****************************************************************************

void Data::calcImpulseResponse()
{
  calcSpectrum(sweepResponse[selectedResponse], responseSpectrum[selectedResponse]);
  calcSpectrum(inverseSourceSignal, tempSpectrum);

  // Multiply both spectra and put the result in "spectrum".
  (*responseSpectrum[selectedResponse]) *= (*tempSpectrum);

  *tempSpectrum = *responseSpectrum[selectedResponse];
  realIFFT(*tempSpectrum, CONVOLVED_SIGNAL_LENGTH_EXPONENT, true);

  // ****************************************************************
  // Copy the time signal into the impulse response track
  // and normalize the amplitude.
  // ****************************************************************

  impulseResponse[selectedResponse]->reset(CONVOLVED_SIGNAL_LENGTH_PT);
  impulseResponse[selectedResponse]->setZero();

  // Determine the absolute maximum value of the impulse response.
  
  int i;
  double max = 0.000000001;
  for (i = 0; i < CONVOLVED_SIGNAL_LENGTH_PT; i++)
  {
    if (fabs(tempSpectrum->re[i]) > max)
    {
      max = fabs(tempSpectrum->re[i]);
    }
  }
  peakNormalizationFactor[selectedResponse] = 1.0 / max;

  for (i = 0; i < CONVOLVED_SIGNAL_LENGTH_PT; i++)
  {
    impulseResponse[selectedResponse]->x[i] = tempSpectrum->re[i] / max;
  }

  // ****************************************************************
  // Calculate the response spectrum again based on the *normalized*
  // impulse response, so that the gain is comparable to the 
  // spectra obtained with calcSmoothedspectrum().
  // ****************************************************************

  ComplexSignal *s = responseSpectrum[selectedResponse];

  // The length of the spectrum must be that of the final convolved signal.
  s->reset(CONVOLVED_SIGNAL_LENGTH_PT);

  // Copy the impulse response into the real part of the spectrum.

  for (i = 0; i < CONVOLVED_SIGNAL_LENGTH_PT; i++)
  {
    s->re[i] = impulseResponse[selectedResponse]->x[i];
    s->im[i] = 0.0;
  }

  realFFT(*s, CONVOLVED_SIGNAL_LENGTH_EXPONENT, false);
}


// ****************************************************************************
/// Calculates a smoothed spectrum by considering only the part of the impulse 
/// response within the selected time window. The first and last 10 ms within 
/// the time window will be tapered to zero with a raised cosine window.
// ****************************************************************************

void Data::calcSmoothedSpectrum()
{
  // The target spectrum is that of the selected response.
  ComplexSignal *s = responseSpectrum[selectedResponse];
  int i;

  // The length of the spectrum must be that of the final convolved signal.
  s->reset(CONVOLVED_SIGNAL_LENGTH_PT);

  // ****************************************************************
  // Copy the (complete) selected sweep response into the real part 
  // of the spectrum.
  // ****************************************************************

  for (i = 0; i < CONVOLVED_SIGNAL_LENGTH_PT; i++)
  {
    s->re[i] = impulseResponse[selectedResponse]->x[i];
    s->im[i] = 0.0;
  }

  // ****************************************************************
  // Set all samples outside the selection to zero.
  // ****************************************************************

  int firstSelectionSample = (int)(selectionMark_s[0] * (double)SAMPLING_RATE);
  int lastSelectionSample = (int)(selectionMark_s[1] * (double)SAMPLING_RATE);
  
  if (firstSelectionSample < 0)
  {
    firstSelectionSample = 0;
  }
  if (firstSelectionSample >= CONVOLVED_SIGNAL_LENGTH_PT)
  {
    firstSelectionSample = CONVOLVED_SIGNAL_LENGTH_PT - 1;
  }

  if (lastSelectionSample < 0)
  {
    lastSelectionSample = 0;
  }
  if (lastSelectionSample >= CONVOLVED_SIGNAL_LENGTH_PT)
  {
    lastSelectionSample = CONVOLVED_SIGNAL_LENGTH_PT - 1;
  }

  if (firstSelectionSample > lastSelectionSample)
  {
    firstSelectionSample = lastSelectionSample;
  }

  for (i = 0; i < firstSelectionSample; i++)
  {
    s->re[i] = 0.0;
  }

  for (i = lastSelectionSample; i < CONVOLVED_SIGNAL_LENGTH_PT; i++)
  {
    s->re[i] = 0.0;
  }

  // ****************************************************************
  // Apply the taper window to the initial and final part of the 
  // selection.
  // ****************************************************************

  int numTaperSamples = (int)(TAPER_LENGTH_S * (double)SAMPLING_RATE);
  double w;
  int k;

  for (i = 0; i < numTaperSamples; i++)
  {
    w = 0.5*(1.0 + cos(M_PI*i / numTaperSamples));

    k = lastSelectionSample - numTaperSamples + 1 + i;
    if ((k >= 0) && (k < CONVOLVED_SIGNAL_LENGTH_PT))
    {
      s->re[k] *= w;
    }

    k = firstSelectionSample + numTaperSamples - 1 - i;
    if ((k >= 0) && (k < CONVOLVED_SIGNAL_LENGTH_PT))
    {
      s->re[k] *= w;
    }
  }

  // ****************************************************************
  // Calculate the FFT.
  // ****************************************************************

  realFFT(*s, CONVOLVED_SIGNAL_LENGTH_EXPONENT, false);
}


// ****************************************************************************
/// Save the spectrum with a resolution of approx. 1 Hz.
// ****************************************************************************

bool Data::saveSpectrum(const wxString &fileName, ComplexSignal *spectrum)
{
  if (fileName.IsEmpty())
  {
    return false;
  }

  ofstream os(fileName.ToStdString());

  if (!os)
  {
    wxMessageBox(wxString("Could not open ") + wxString(fileName) + wxString(" for writing."),
      wxString("Error!"));
    return false;
  }

  // ****************************************************************
  // Write the samples of the signals.
  // ****************************************************************

  const double AMPLITUDE_SCALE = 1.0;
  double F0 = (double)SAMPLING_RATE / (double)spectrum->N;
  int stepSize = (int)(1.0 / F0);
  if (stepSize < 1)
  {
    stepSize = 1;
  }
  // Save the spectrum up to 20 kHz.
  int maxSample = (int)(20000.0 / F0);
  if (maxSample >= spectrum->N)
  {
    maxSample = spectrum->N - 1;
  }

  os << "freq_Hz  magnitude  phase_rad" << endl;

  // Average stepSize spectral samples for each output value.
  // This is a low-pass filter needed to avoid aliasing.

  int i, k;
  Complex c;
  double freq_Hz;
  int pos;

  for (i = 0; i < maxSample; i += stepSize)
  {
    c = Complex(0.0, 0.0);
    for (k = 0; k < stepSize; k++)
    {
      pos = i - stepSize / 2 + k;
      if (pos < 0) { pos = 0; }
      if (pos > spectrum->N) { pos = spectrum->N - 1; }
      c += spectrum->getValue(pos);
    }
    c /= (double)stepSize;
    
    freq_Hz = (double)i * F0;
    os << freq_Hz << " "
      << std::abs(c) * AMPLITUDE_SCALE << " "
      << std::arg(c) << " "
      << endl;
  }


  // ****************************************************************
  // Close the file.
  // ****************************************************************

  os.close();

  return true;
}


// ****************************************************************************
/// Constructor.
// ****************************************************************************

Data::Data()
{
  // Do nothing. Initialization is done in init().
}

// ****************************************************************************
