// ****************************************************************************
// GlottalImageExplorer.
// Copyright (C) 2015-2016 Peter Birkholz.
// This program is free and open-source software.
// ****************************************************************************

#include "ImageProc.h"

float ImageProc::xMatrix[ImageProc::NUM_PIXELS] = { 0.0 };
float ImageProc::yMatrix[ImageProc::NUM_PIXELS] = { 0.0 };
float ImageProc::gradientMagnitude[ImageProc::NUM_PIXELS] = { 0.0 };


// ****************************************************************************
/// Calculates the edge image from the source image.
// ****************************************************************************

wxImage ImageProc::getEdgeImage(wxImage *sourceImage)
{
  const int NUM_PIXELS = Film::WIDTH * Film::HEIGHT;
  float origMatrix[NUM_PIXELS];
  float xFilter[] = {-1, 0, 1, -2, 0, 2, -1, 0, 1};
  float yFilter[] = {-1, -2, -1, 0, 0, 0, 1, 2, 1};
  int i;

  // Calculate the gray values of the original image.
  imageToMatrix(sourceImage, origMatrix);

  // Take the derivative with respect to x and y.
  filterMatrix3x3(origMatrix, xMatrix, xFilter);
  filterMatrix3x3(origMatrix, yMatrix, yFilter);

  // ****************************************************************
  // Get the magnitude of the gradient.
  // ****************************************************************

  float radicand;
  float x0;
  for (i=0; i < NUM_PIXELS; i++)
  {
    radicand = xMatrix[i]*xMatrix[i] + yMatrix[i]*yMatrix[i];
    // First approximation of x = sqrt(x*x + y*y) by the sum of the absolute values.
    x0 = fabs(xMatrix[i]) + fabs(yMatrix[i]);
    if (x0 < (float)0.000001)
    {
      x0 = (float)0.000001;
    }
    // Make the first recursion step of the heron algorithm.
    gradientMagnitude[i] = 0.5*(x0 + radicand/x0);
  }

  // ****************************************************************

  normalizeMatrix(gradientMagnitude, false);

  // Create and return the target RGB-image.
  return matrixToImage(gradientMagnitude, false);
}


// ****************************************************************************
/// Calculate the magnitude of the gradient along the normals of the principal
/// glottis axis.
// ****************************************************************************

wxImage ImageProc::getAlignedEdgeImage(wxImage *sourceImage, double glottisAxisAngle_deg)
{
  double vx = cos(glottisAxisAngle_deg*M_PI / 180.0);
  double vy = sin(glottisAxisAngle_deg*M_PI / 180.0);

  const int NUM_PIXELS = Film::WIDTH * Film::HEIGHT;
  float origMatrix[NUM_PIXELS];
  float xFilter[] = {-1, 0, 1, -2, 0, 2, -1, 0, 1};
  float yFilter[] = {-1, -2, -1, 0, 0, 0, 1, 2, 1};
  int i;

  // Calculate the gray values of the original image.
  imageToMatrix(sourceImage, origMatrix);

  // Take the derivative with respect to x and y.
  filterMatrix3x3(origMatrix, xMatrix, xFilter);
  filterMatrix3x3(origMatrix, yMatrix, yFilter);

  // Take the scalar product of the gradient field and the normal vector
  
  for (i=0; i < NUM_PIXELS; i++)
  {
    gradientMagnitude[i] = xMatrix[i]*vx + yMatrix[i]*vy;
  }

  // ****************************************************************

  normalizeMatrix(gradientMagnitude, true);

  // Create and return the target RGB-image.
  return matrixToImage(gradientMagnitude, true);
}


// ****************************************************************************
/// Segments the glottis in the given source image.
/// As a result, matrix[Film::WIDTH*FILM::HEIGHT] contains a "0" at all pixels 
/// that are not part of the glottis and "1" for all pixels marked during the 
/// seeded region growing.
/// From this matrix, the actual glottis is segmented by searching in each scan 
/// line for the leftmost and the rightmost marked pixel in the matrix.
/// The corresponding x-positions of the pixels will be saved in
/// leftContour[Film::HEIGHT] and rightContour[Film::HEIGHT].
///
/// The return value is the number of glottis pixels between (and including)
/// the left and right contour. It is NOT necessarily identical with the number 
/// of pixels marked in the matrix.
// ****************************************************************************

int ImageProc::segmentGlottis(wxImage *sourceImage, int segmentMatrix[], 
  int leftContour[], int rightContour[], Data::SrgData *srgData)
{
  // Initialize the resulting glottis matrix with zeros.
  int i;
  for (i = 0; i < NUM_PIXELS; i++)
  {
    segmentMatrix[i] = 0;
  }

  // Calc. the gray value matrix.
  float grayValues[NUM_PIXELS];
  imageToMatrix(sourceImage, grayValues);

  int rowThreshold[Film::HEIGHT];
  getRowThresholds(srgData, rowThreshold);


  // ****************************************************************
  // Prepare the region growing.
  // ****************************************************************

  int offset = 0;

  // The seed points are parts of the glottis, if the corresponding
  // gray values are above threshold.
  for (i = 0; i < Data::NUM_SEEDS; i++)
  {
    offset = srgData->seed[i].y * Film::WIDTH + srgData->seed[i].x;
    if (grayValues[offset] < rowThreshold[srgData->seed[i].y])
    {
      segmentMatrix[offset] = 1;
    }
  }

  // ****************************************************************
  // Do the region growing in multiple passes from top-left to
  // bottom-right and back.
  // ****************************************************************

  int x, y;
  int numRegionPixels = 0;
  int oldNumRegionPixels = 0;

  do
  {
    // Pass from top left to bottom right

    for (x = 1; x < Film::WIDTH - 1; x++)
    {
      for (y = 1; y < Film::HEIGHT - 1; y++)
      {
        if (segmentMatrix[y*Film::WIDTH + x] > 0)
        {
          // Check the 8 neigbors!
          offset = (y - 1) * Film::WIDTH + (x - 1);
          if (grayValues[offset] < rowThreshold[y-1]) { segmentMatrix[offset] = 1; }
          offset = (y - 1) * Film::WIDTH + (x);
          if (grayValues[offset] < rowThreshold[y-1]) { segmentMatrix[offset] = 1; }
          offset = (y - 1) * Film::WIDTH + (x + 1);
          if (grayValues[offset] < rowThreshold[y-1]) { segmentMatrix[offset] = 1; }
          offset = (y) * Film::WIDTH + (x + 1);
          if (grayValues[offset] < rowThreshold[y]) { segmentMatrix[offset] = 1; }
          offset = (y + 1) * Film::WIDTH + (x + 1);
          if (grayValues[offset] < rowThreshold[y+1]) { segmentMatrix[offset] = 1; }
          offset = (y + 1) * Film::WIDTH + (x);
          if (grayValues[offset] < rowThreshold[y+1]) { segmentMatrix[offset] = 1; }
          offset = (y + 1) * Film::WIDTH + (x - 1);
          if (grayValues[offset] < rowThreshold[y+1]) { segmentMatrix[offset] = 1; }
          offset = (y) * Film::WIDTH + (x - 1);
          if (grayValues[offset] < rowThreshold[y]) { segmentMatrix[offset] = 1; }
        }
      }
    }

    // Pass from bottom right to top left

    for (x = Film::WIDTH - 2; x >= 1; x--)
    {
      for (y = Film::HEIGHT - 2; y >= 1; y--)
      {
        if (segmentMatrix[y*Film::WIDTH + x] > 0)
        {
          // Check the 8 neigbors!
          offset = (y - 1) * Film::WIDTH + (x - 1);
          if (grayValues[offset] < rowThreshold[y-1]) { segmentMatrix[offset] = 1; }
          offset = (y - 1) * Film::WIDTH + (x);
          if (grayValues[offset] < rowThreshold[y-1]) { segmentMatrix[offset] = 1; }
          offset = (y - 1) * Film::WIDTH + (x + 1);
          if (grayValues[offset] < rowThreshold[y-1]) { segmentMatrix[offset] = 1; }
          offset = (y)* Film::WIDTH + (x + 1);
          if (grayValues[offset] < rowThreshold[y]) { segmentMatrix[offset] = 1; }
          offset = (y + 1) * Film::WIDTH + (x + 1);
          if (grayValues[offset] < rowThreshold[y+1]) { segmentMatrix[offset] = 1; }
          offset = (y + 1) * Film::WIDTH + (x);
          if (grayValues[offset] < rowThreshold[y+1]) { segmentMatrix[offset] = 1; }
          offset = (y + 1) * Film::WIDTH + (x - 1);
          if (grayValues[offset] < rowThreshold[y+1]) { segmentMatrix[offset] = 1; }
          offset = (y)* Film::WIDTH + (x - 1);
          if (grayValues[offset] < rowThreshold[y]) { segmentMatrix[offset] = 1; }
        }
      }
    }

    // Count the glottis pixels in the segmentation image and compare
    // them with the previous value.
    oldNumRegionPixels = numRegionPixels;

    numRegionPixels = 0;
    for (x = 0; x < Film::WIDTH; x++)
    {
      for (y = 0; y < Film::HEIGHT; y++)
      {
        if (segmentMatrix[y*Film::WIDTH + x] > 0) { numRegionPixels++; }
      }
    }
  } 
  while (numRegionPixels > oldNumRegionPixels);


  // ****************************************************************
  // For each pixel row, search for the left and right border
  // of the glottis.
  // ****************************************************************

  int numGlottisPixels = 0;

  for (y = 0; y < Film::HEIGHT; y++)
  {
    leftContour[y] = 0;
    while ((leftContour[y] < Film::WIDTH) &&
      (segmentMatrix[y*Film::WIDTH + leftContour[y]] == 0))
    {
      leftContour[y]++;
    }

    rightContour[y] = Film::WIDTH - 1;
    while ((rightContour[y] >= 0) &&
      (segmentMatrix[y*Film::WIDTH + rightContour[y]] == 0))
    {
      rightContour[y]--;
    }

    // Add up the number of actual glottis pixels.

    if ((leftContour[y] < Film::WIDTH) && (rightContour[y] >= 0) &&
      (leftContour[y] <= rightContour[y]))
    {
      numGlottisPixels += (rightContour[y] - leftContour[y]) + 1;
    }
    else
    {
      // Invalid border.
      leftContour[y] = 0;
      rightContour[y] = 0;
    }
  }

  // Return the number of actual glottis pixels.
  return numGlottisPixels;
}


// ****************************************************************************
/// Calculates the gray value threshold for every row of the image for
/// glottis segmentation based on seeded region growing.
/// The return value rowThreshold[] must contain Film::HEIGHT elements.
// ****************************************************************************

void ImageProc::getRowThresholds(Data::SrgData *srgData, int rowThreshold[])
{
  int y, i;

  // Init with default values.

  for (y = 0; y < srgData->thresholdY[0]; y++)
  {
    rowThreshold[y] = 0;
  }

  // ****************************************************************

  for (y = 0; y < srgData->thresholdY[0]; y++)
  {
    rowThreshold[y] = srgData->thresholdValue[0];
  }

  int y0, y1;
  int value0, value1;
  int deltaY;

  for (i = 0; i < Data::NUM_THRESHOLD_POINTS - 1; i++)
  {
    y0 = srgData->thresholdY[i];
    y1 = srgData->thresholdY[i + 1];
    deltaY = y1 - y0;
    if (deltaY < 1)
    {
      deltaY = 1;
    }

    value0 = srgData->thresholdValue[i];
    value1 = srgData->thresholdValue[i + 1];

    for (y = y0; y <= y1; y++)
    {
      rowThreshold[y] = value0 + (value1 - value0)*(y - y0) / deltaY;
    }
  }

  for (y = srgData->thresholdY[Data::NUM_THRESHOLD_POINTS - 1]; y < Film::HEIGHT; y++)
  {
    rowThreshold[y] = srgData->thresholdValue[Data::NUM_THRESHOLD_POINTS - 1];
  }
}


// ****************************************************************************
/// Converts the given RGB-image into a grayscale matrix.
// ****************************************************************************

void ImageProc::imageToMatrix(wxImage *image, float matrix[])
{
  const int NUM_PIXELS = Film::WIDTH * Film::HEIGHT;
  int i;
  float red, green, blue;
  unsigned char *p = image->GetData();

  for (i=0; i < NUM_PIXELS; i++)
  {
    red = *p++;
    green = *p++;
    blue = *p++;
    matrix[i] = 0.299*red + 0.587*green + 0.114*blue;
  }

  removeHighlights(matrix);
}


// ****************************************************************************
/// Converts the given grayscale matrix to a grayscale RGB-image.
// ****************************************************************************

wxImage ImageProc::matrixToImage(float matrix[], bool bipolar)
{
  const int NUM_PIXELS = Film::WIDTH * Film::HEIGHT;
  wxImage image(Film::WIDTH, Film::HEIGHT);
  unsigned char *p = image.GetData();
  unsigned char c;
  int i;

  if (bipolar)
  {
    for (i=0; i < NUM_PIXELS; i++)
    {
      c = 128 + (unsigned char)(matrix[i]*127.0);
      *p++ = c;
      *p++ = c;
      *p++ = c;
    }
  }
  else
  {
    for (i=0; i < NUM_PIXELS; i++)
    {
      c = (unsigned char)(matrix[i]*255.0);
      *p++ = c;
      *p++ = c;
      *p++ = c;
    }
  }

  return image;
}


// ****************************************************************************
/// Filters the given source matrix with the 3x3 pixel filter kernel and
/// returns the filtered target matrix.
// ****************************************************************************

void ImageProc::filterMatrix3x3(float sourceMatrix[], float targetMatrix[], float filter3x3[])
{
  const int W = Film::WIDTH;
  const int H = Film::HEIGHT;
  int x, y;
  float sum;
  float *p = sourceMatrix;
  int offset;

  // ****************************************************************
  // Calculate all pixels except the border pixels.
  // ****************************************************************

  for (y=1; y < H-1; y++)
  {
    // Offset of the second pixel in the current row
    offset = y*W + 1;
    for (x=1; x < W-1; x++)
    {
      sum = 
        filter3x3[0] * p[offset-W-1] +
        filter3x3[1] * p[offset-W] +
        filter3x3[2] * p[offset-W+1] +
        filter3x3[3] * p[offset-1] +
        filter3x3[4] * p[offset] +
        filter3x3[5] * p[offset+1] +
        filter3x3[6] * p[offset+W-1] +
        filter3x3[7] * p[offset+W] +
        filter3x3[8] * p[offset+W+1];

      targetMatrix[offset] = sum;
      offset++;
    }
  }

  // ****************************************************************
  // Calculate the border pixels.
  // ****************************************************************

  for (x=1; x < W-1; x++)
  {
    targetMatrix[x] = targetMatrix[x + W];
    targetMatrix[(H-1)*W + x] = targetMatrix[(H-2)*W + x];
  }

  for (y=1; y < H-1; y++)
  {
    targetMatrix[y*W] = targetMatrix[y*W + 1];
    targetMatrix[y*W + W - 2] = targetMatrix[y*W + W - 1];
  }

  // ****************************************************************
  // Calculate the corner pixels.
  // ****************************************************************

  targetMatrix[0] = targetMatrix[1];
  targetMatrix[W-1] = targetMatrix[W-2];
  targetMatrix[(H-1)*W] = targetMatrix[(H-1)*W+1];
  targetMatrix[W*H-1] = targetMatrix[W*H-2];
}


// ****************************************************************************
/// Normalize all data to the interval [0, +1] or [-1, +1].
// ****************************************************************************

void ImageProc::normalizeMatrix(float matrix[], bool bipolar)
{
  // Skip pixels in the first and last rows, because they are
  // aretefacts in the images that we have...
  const int NUM_SKIP_PIXELS = Film::WIDTH * 10;

  const int N = Film::WIDTH * Film::HEIGHT;
  float reference = 0.0;
  int i;

  // Find the reference (extreme) value

  for (i = NUM_SKIP_PIXELS; i < N - NUM_SKIP_PIXELS; i++)
  {
    if (matrix[i] > reference)
    {
      reference = matrix[i];
    }
    if ((bipolar) && (-matrix[i] > reference))
    {
      reference = -matrix[i];
    }
  }

  if (fabs(reference) < (float)0.000001)
  {
    reference = (float)0.000001;
  }
  float invRef = 1.0 / reference;

  // Divide all matrix values by the reference value

  for (i=0; i < N; i++)
  {
    matrix[i] *= invRef;

    // Safety check, because we skipped some pixels during
    // the determination of the reference value.
    if (matrix[i] > 1.0)
    {
      matrix[i] = 1.0;
    }
    if (matrix[i] < -1.0)
    {
      matrix[i] = -1.0;
    }
  }
}

// ****************************************************************************
/// Removes the highlights from a grayscale matrix, because they are
/// disturbing for edge detection.
// ****************************************************************************

void ImageProc::removeHighlights(float matrix[])
{
  const int N = Film::WIDTH * Film::HEIGHT;
  int i;
  float maximum = 0.0;

  // Find the maximum value

  for (i=0; i < N; i++)
  {
    if (matrix[i] > maximum)
    {
      maximum = matrix[i];
    }
  }

  // Remove all bright image parts and strech the color values inbetween.
  const float factor = (float)1.5;

  for (i=0; i < N; i++)
  {
    matrix[i]*= factor;
    if (matrix[i] > maximum)
    {
      matrix[i] = maximum;
    }
  }
}


// ****************************************************************************
