// ****************************************************************************
// GlottalImageExplorer.
// Copyright (C) 2015-2016 Peter Birkholz.
// This program is free and open-source software.
// ****************************************************************************

#include "FilmPicture.h"
#include "ImageProc.h"
#include <wx/rawbmp.h>

// ****************************************************************************
// IDs.
// ****************************************************************************

static const int IDM_USER_DEFINED_SEGMENTATION = 8000;
static const int IDM_SHOW_SEEDS = 8001;
static const int IDM_SHOW_THRESHOLD = 8002;
static const int IDM_TO_CURRENT_IMAGE = 8003;
static const int IDM_TO_SIDE_IMAGE_1 = 8004;
static const int IDM_TO_SIDE_IMAGE_2 = 8005;


// ****************************************************************************
// The event table.
// ****************************************************************************

BEGIN_EVENT_TABLE(FilmPicture, BasicPicture)
  EVT_MOUSE_EVENTS(OnMouseEvent)
  EVT_MENU(IDM_USER_DEFINED_SEGMENTATION, OnUserDefinedSegmentation)
  EVT_MENU(IDM_SHOW_SEEDS, OnShowSeeds)
  EVT_MENU(IDM_SHOW_THRESHOLD, OnShowThreshold)
  EVT_MENU(IDM_TO_CURRENT_IMAGE, OnToCurrentImage)
  EVT_MENU(IDM_TO_SIDE_IMAGE_1, OnToSideImage1)
  EVT_MENU(IDM_TO_SIDE_IMAGE_2, OnToSideImage2)
  END_EVENT_TABLE()


// ****************************************************************************
/// Construcor.
/// pictureIndex is the index of the picture widget, i.e. 0, 1, or 2.
// ****************************************************************************

FilmPicture::FilmPicture(wxWindow *parent, int pictureIndex) : BasicPicture(parent)
{
  image = new wxImage(Film::WIDTH, Film::HEIGHT);
  this->pictureIndex = pictureIndex;
  data = Data::getInstance();

  // Create the context menu

  contextMenu = new wxMenu();
  contextMenu->AppendCheckItem(IDM_USER_DEFINED_SEGMENTATION, "User defined segmentation");
  contextMenu->AppendCheckItem(IDM_SHOW_SEEDS, "Show seeds");
  contextMenu->AppendCheckItem(IDM_SHOW_THRESHOLD, "Show threshold");
  contextMenu->AppendSeparator();
  contextMenu->Append(IDM_TO_CURRENT_IMAGE, "To current image");
  contextMenu->Append(IDM_TO_SIDE_IMAGE_1, "To side image 1");
  contextMenu->Append(IDM_TO_SIDE_IMAGE_2, "To side image 2");

  menuX = 0;
  menuY = 0;
  lastMx = 0;
  lastMy = 0;
  selectedSeed = -1;
  selectedThresholdPoint = -1;

  if (pictureIndex == 0)
  {
    showSeeds = true;
    showThreshold = true;
  }
  else
  {
    showSeeds = false;
    showThreshold = false;
  }
}


// ****************************************************************************
/// Draws the picture.
// ****************************************************************************

void FilmPicture::draw(wxDC &dc)
{
  static const int TEXT_HEIGHT = 16;
  int i;
  int x, y;

  int areaWidth, areaHeight;
  this->GetSize(&areaWidth, &areaHeight);

  // Clear the background
  dc.SetBackground(*wxWHITE_BRUSH);
  dc.Clear();

  // ****************************************************************
  // Scale the image, convert it to a bitmap and draw it.
  // ****************************************************************

  if ((pictureIndex < 0) || (pictureIndex >= Data::NUM_DISPLAYED_FRAMES))
  {
    return;
  }

  int frameIndex = data->getCurrFrameIndex(pictureIndex);
  if (frameIndex == -1)
  {
    return;
  }

  *image = data->film->getFrame(frameIndex);

  wxImage tempImage;
  
  if (data->imageType == Data::COLOR_IMAGE)
  {
    tempImage = *image;
  }
  else
  if (data->imageType == Data::GRAYSCALE_IMAGE)
  {
    tempImage = image->ConvertToGreyscale();
  }
  else
  if (data->imageType == Data::EDGE_IMAGE)
  {
    tempImage = ImageProc::getEdgeImage(image);
  }
  else
  if (data->imageType == Data::ALIGNED_EDGE_IMAGE)
  {
    tempImage = ImageProc::getAlignedEdgeImage(image, 4.0);
  }
  else
  // Error case !!!!!
  {
    tempImage = image->ConvertToGreyscale();
  }

  // ****************************************************************
  // Draw the segmented area in the image.
  // ****************************************************************

  wxBitmap b(tempImage);
  wxNativePixelData pixelData(b);
  wxNativePixelData::Iterator p(pixelData);
  int segmentMatrix[Film::WIDTH * Film::HEIGHT];
  int leftContour[Film::HEIGHT];
  int rightContour[Film::HEIGHT];
  int numGlottisPixels = 0;

  if ((data->showGlottisArea) || (data->showGlottisEdge))
  {
    numGlottisPixels = ImageProc::segmentGlottis(image, segmentMatrix, 
      leftContour, rightContour, &data->srgData[frameIndex]);

    for (x = 0; x < Film::WIDTH; x++)
    {
      for (y = 0; y < Film::HEIGHT; y++)
      {
        if ((segmentMatrix[y*Film::WIDTH + x] > 0) && (data->showGlottisArea))
        {
          p.MoveTo(pixelData, x, y);
          p.Red() = 255;
          p.Green() = 0;
          p.Blue() = 255;
        }
      }
    }

    for (y = 0; y < Film::HEIGHT; y++)
    {
      if (leftContour[y] != 0)
      {
        p.MoveTo(pixelData, leftContour[y], y);
        p.Red() = 255;
        p.Green() = 255;
        p.Blue() = 0;
      }

      if (rightContour[y] != 0)
      {
        p.MoveTo(pixelData, rightContour[y], y);
        p.Red() = 255;
        p.Green() = 255;
        p.Blue() = 0;
      }
    }
  }

  tempImage = b.ConvertToImage();

  // ****************************************************************
  // Scale and draw the image.
  // ****************************************************************

  tempImage.Rescale(this->GetSize().GetWidth(), this->GetSize().GetHeight());
  wxBitmap bitmap(tempImage);
  dc.DrawBitmap(bitmap, 0, 0);

  // ****************************************************************
  // If this frame has a user defined segmentation, draw a
  // border around it.
  // ****************************************************************

  if (data->srgData[frameIndex].isUserDefinedSegmentation)
  {
    dc.SetBrush(*wxTRANSPARENT_BRUSH);
    dc.SetPen(wxPen(*wxRED, 3)); // wxPen(wxColor(255, 255, 0), 3));

    dc.DrawRectangle(1, 1, areaWidth-1, areaHeight-1);
  }

  // ****************************************************************
  // Draw the seeds.
  // ****************************************************************

  if (showSeeds)
  {
    dc.SetBackgroundMode(wxTRANSPARENT);

    for (i = 0; i < (int)Data::NUM_SEEDS; i++)
    {
      x = data->srgData[frameIndex].seed[i].x * areaWidth / Film::WIDTH;
      y = data->srgData[frameIndex].seed[i].y * areaHeight / Film::HEIGHT;

      if (data->srgData[frameIndex].isUserDefinedSegmentation == false)
      {
        dc.SetBrush(*wxLIGHT_GREY_BRUSH);
        dc.SetPen(*wxLIGHT_GREY_PEN);
        dc.SetTextForeground(*wxLIGHT_GREY);
      }
      else
      if (i == selectedSeed)
      {
        dc.SetBrush(*wxRED_BRUSH);
        dc.SetPen(*wxRED_PEN);
        dc.SetTextForeground(*wxBLACK);
      }
      else
      {
        dc.SetBrush(*wxWHITE_BRUSH);
        dc.SetPen(*wxWHITE_PEN);
        dc.SetTextForeground(*wxBLACK);
      }

      dc.DrawCircle(x, y, 4);
      dc.DrawText(wxString::Format("S%d", i), x+3, y-3-TEXT_HEIGHT);
    }
  }

  // ****************************************************************
  // Draw the threshold points and line.
  // ****************************************************************

  if (showThreshold)
  {
    // Draw the lines.

    int rowThreshold[Film::HEIGHT];
    ImageProc::getRowThresholds(&data->srgData[frameIndex], rowThreshold);
    int x0, x1, y0, y1;

    if (data->srgData[frameIndex].isUserDefinedSegmentation == false)
    {
      dc.SetPen(*wxLIGHT_GREY_PEN);
    }
    else
    {
      dc.SetPen(*wxWHITE_PEN);
    }

    for (i = 0; i < Film::HEIGHT - 1; i++)
    {
      x0 = rowThreshold[i] * areaWidth / 256;
      y0 = i * areaHeight / Film::HEIGHT;

      x1 = rowThreshold[i + 1] * areaWidth / 256;
      y1 = (i + 1) * areaHeight / Film::HEIGHT;

      dc.DrawLine(x0, y0, x1, y1);
    }

    // Draw the control points.

    dc.SetBackgroundMode(wxTRANSPARENT);

    for (i = 0; i < (int)Data::NUM_THRESHOLD_POINTS; i++)
    {
      x = data->srgData[frameIndex].thresholdValue[i] * areaWidth / Film::WIDTH;
      y = data->srgData[frameIndex].thresholdY[i] * areaHeight / Film::HEIGHT;

      // Draw connecting dashed lines between seeds and threshold points?

      if ((showSeeds) && (data->seedThresholdCoupling))
      {
        x1 = data->srgData[frameIndex].seed[i].x * areaWidth / Film::WIDTH;
        if (data->srgData[frameIndex].isUserDefinedSegmentation == false)
        {
          dc.SetPen(wxPen(*wxLIGHT_GREY, 1, wxDOT));
        }
        else
        {
          dc.SetPen(wxPen(*wxWHITE, 1, wxDOT));
        }
        dc.DrawLine(x, y, x1, y);
      }

      // Draw the threshold point circles.

      if (data->srgData[frameIndex].isUserDefinedSegmentation == false)
      {
        dc.SetBrush(*wxLIGHT_GREY_BRUSH);
        dc.SetPen(*wxLIGHT_GREY_PEN);
        dc.SetTextForeground(*wxLIGHT_GREY);
      }
      else
      if (i == selectedThresholdPoint)
      {
        dc.SetBrush(*wxRED_BRUSH);
        dc.SetPen(*wxRED_PEN);
        dc.SetTextForeground(*wxBLACK);
      }
      else
      {
        dc.SetBrush(*wxWHITE_BRUSH);
        dc.SetPen(*wxWHITE_PEN);
        dc.SetTextForeground(*wxBLACK);
      }

      dc.DrawCircle(x, y, 4);
      dc.DrawText(wxString::Format("%d", data->srgData[frameIndex].thresholdValue[i]), x + 3, y - 3 - TEXT_HEIGHT);
    }
  }

  // ****************************************************************
  // Draw the frame number and glottis area.
  // ****************************************************************

  dc.SetTextForeground(*wxBLACK);
  dc.SetBackground(*wxWHITE_BRUSH);
  dc.SetBackgroundMode(wxSOLID);

  wxString st = wxString::Format("Frame %d", frameIndex);
  dc.DrawText(st, 3, 3);

  if ((data->showGlottisArea) || (data->showGlottisEdge))
  {
    wxString st = wxString::Format("Glottis area: %d", numGlottisPixels);
    dc.DrawText(st, 3, areaHeight - 20);
  }

}


// ****************************************************************************
// ****************************************************************************

void FilmPicture::OnMouseEvent(wxMouseEvent &event)
{
  int areaWidth, areaHeight;
  this->GetSize(&areaWidth, &areaHeight);

  int mx = event.GetX();
  int my = event.GetY();
  int frameIndex = data->getCurrFrameIndex(pictureIndex);

  // ****************************************************************
  // The mouse is entering the window.
  // ****************************************************************

  if (event.Entering())
  {
    // Automatically set the focus to this window, so that
    // mouse wheel events are properly received !!
    this->SetFocus();

    selectedSeed = -1;
    selectedThresholdPoint = -1;
    lastMx = mx;
    lastMy = my;
    return;
  }

  // ****************************************************************
  // This a button-up event.
  // ****************************************************************

  if (event.ButtonUp())
  {
    selectedSeed = -1;
    selectedThresholdPoint = -1;
    lastMx = mx;
    lastMy = my;

    this->Refresh();
    return;
  }

  // ****************************************************************
  // The left mouse button changed to down.
  // ****************************************************************

  if ((event.LeftDown()) && (frameIndex != -1) && (data->srgData[frameIndex].isUserDefinedSegmentation))
  {
    static const int CATCH_RANGE = 5;
    int i;
    int x, y;

    // Did the user click on an existing seed ?

    selectedSeed = -1;
    for (i = 0; i < Data::NUM_SEEDS; i++)
    {
      x = data->srgData[frameIndex].seed[i].x * areaWidth / Film::WIDTH;
      y = data->srgData[frameIndex].seed[i].y * areaHeight / Film::HEIGHT;

      if ((abs(mx - x) < CATCH_RANGE) && (abs(my - y) < CATCH_RANGE))
      {
        selectedSeed = i;
      }
    }

    selectedThresholdPoint = -1;
    if (selectedSeed == -1)
    {
      for (i = 0; i < Data::NUM_THRESHOLD_POINTS; i++)
      {
        x = data->srgData[frameIndex].thresholdValue[i] * areaWidth / Film::WIDTH;
        y = data->srgData[frameIndex].thresholdY[i] * areaHeight / Film::HEIGHT;

        if ((abs(mx - x) < CATCH_RANGE) && (abs(my - y) < CATCH_RANGE))
        {
          selectedThresholdPoint = i;
        }
      }
    }

    lastMx = mx;
    lastMy = my;

    this->Refresh();
    return;
  }

  // ****************************************************************
  // The right mouse button changed to down.
  // ****************************************************************

  if (event.RightDown())
  {
    if (frameIndex != -1)
    {
      menuX = mx;
      menuY = my;

      contextMenu->Check(IDM_USER_DEFINED_SEGMENTATION, data->srgData[frameIndex].isUserDefinedSegmentation);
      contextMenu->Check(IDM_SHOW_SEEDS, showSeeds);
      contextMenu->Check(IDM_SHOW_THRESHOLD, showThreshold);

      if (pictureIndex == 0)
      {
        contextMenu->Enable(IDM_TO_CURRENT_IMAGE, false);
        contextMenu->Enable(IDM_TO_SIDE_IMAGE_1, true);
        contextMenu->Enable(IDM_TO_SIDE_IMAGE_2, true);
      }
      else
      {
        contextMenu->Enable(IDM_TO_CURRENT_IMAGE, true);
        contextMenu->Enable(IDM_TO_SIDE_IMAGE_1, false);
        contextMenu->Enable(IDM_TO_SIDE_IMAGE_2, false);
      }

      PopupMenu(contextMenu);
      return;
    }
  }


  // ****************************************************************
  // The mouse is dragged (with one or more mouse buttons pressed).
  // ****************************************************************

  if (event.Dragging())
  {
    if (areaWidth < 1)
    {
      areaWidth = 1;
    }
    if (areaHeight < 1)
    {
      areaHeight = 1;
    }

    if (selectedSeed != -1)
    {
      data->srgData[frameIndex].seed[selectedSeed].x = mx * Film::WIDTH / areaWidth;
      data->srgData[frameIndex].seed[selectedSeed].y = my * Film::HEIGHT / areaHeight;

      // The seeds' y-coordinates must be ordered like
      // their indices.

      if ((selectedSeed > 0) &&
        (data->srgData[frameIndex].seed[selectedSeed].y <=
        data->srgData[frameIndex].seed[selectedSeed - 1].y))
      {
        data->srgData[frameIndex].seed[selectedSeed].y =
          data->srgData[frameIndex].seed[selectedSeed - 1].y + 1;
      }

      if ((selectedSeed < Data::NUM_SEEDS - 1) &&
        (data->srgData[frameIndex].seed[selectedSeed].y >=
        data->srgData[frameIndex].seed[selectedSeed + 1].y))
      {
        data->srgData[frameIndex].seed[selectedSeed].y =
          data->srgData[frameIndex].seed[selectedSeed + 1].y - 1;
      }

      if (data->seedThresholdCoupling)
      {
        data->srgData[frameIndex].thresholdY[selectedSeed] =
          data->srgData[frameIndex].seed[selectedSeed].y;
      }
      
    }

    if (selectedThresholdPoint != -1)
    {
      data->srgData[frameIndex].thresholdValue[selectedThresholdPoint] = mx * 256 / areaWidth;
      data->srgData[frameIndex].thresholdY[selectedThresholdPoint] = my * Film::HEIGHT / areaHeight;

      // The threshold points' y-coordinates must be ordered like
      // their indices.

      if ((selectedThresholdPoint > 0) &&
        (data->srgData[frameIndex].thresholdY[selectedThresholdPoint] <=
        data->srgData[frameIndex].thresholdY[selectedThresholdPoint - 1]))
      {
        data->srgData[frameIndex].thresholdY[selectedThresholdPoint] =
          data->srgData[frameIndex].thresholdY[selectedThresholdPoint - 1] + 1;
      }

      if ((selectedThresholdPoint < Data::NUM_THRESHOLD_POINTS - 1) &&
        (data->srgData[frameIndex].thresholdY[selectedThresholdPoint] >=
        data->srgData[frameIndex].thresholdY[selectedThresholdPoint + 1]))
      {
        data->srgData[frameIndex].thresholdY[selectedThresholdPoint] =
          data->srgData[frameIndex].thresholdY[selectedThresholdPoint + 1] - 1;
      }

      // y-Coordinate must be same as that of the seeds ?

      if (data->seedThresholdCoupling)
      {
        data->srgData[frameIndex].thresholdY[selectedThresholdPoint] =
          data->srgData[frameIndex].seed[selectedThresholdPoint].y;
      }
    }

    data->interpolateSrgData();

    // Refreh not only this image but also the others with the same parent widget!
    this->GetParent()->Refresh();
  }

  lastMx = mx;
  lastMy = my;
}


// ****************************************************************************
// ****************************************************************************

void FilmPicture::OnUserDefinedSegmentation(wxCommandEvent &event)
{
  int frameIndex = data->getCurrFrameIndex(pictureIndex);
  if (frameIndex != -1)
  {
    data->srgData[frameIndex].isUserDefinedSegmentation = !data->srgData[frameIndex].isUserDefinedSegmentation;
    if (data->srgData[frameIndex].isUserDefinedSegmentation)
    {
      showSeeds = true;
      showThreshold = true;
    }
    data->interpolateSrgData();
    // Refreh not only this image but also the others with the same parent widget!
    this->GetParent()->Refresh();
  }
}


// ****************************************************************************
// ****************************************************************************

void FilmPicture::OnShowSeeds(wxCommandEvent &event)
{
  showSeeds = !showSeeds;
  this->Refresh();
}


// ****************************************************************************
// ****************************************************************************

void FilmPicture::OnShowThreshold(wxCommandEvent &event)
{
  showThreshold = !showThreshold;
  this->Refresh();
}

// ****************************************************************************
// ****************************************************************************

void FilmPicture::OnToCurrentImage(wxCommandEvent &event)
{
  data->frameIndex[0] = data->frameIndex[pictureIndex];
  // Refresh all images of the parent!
  this->GetParent()->Refresh();
}

// ****************************************************************************
// ****************************************************************************

void FilmPicture::OnToSideImage1(wxCommandEvent &event)
{
  data->frameIndex[1] = data->frameIndex[0];
  // Refresh all images of the parent!
  this->GetParent()->Refresh();
}

// ****************************************************************************
// ****************************************************************************

void FilmPicture::OnToSideImage2(wxCommandEvent &event)
{
  data->frameIndex[2] = data->frameIndex[0];
  // Refresh all images of the parent!
  this->GetParent()->Refresh();
}

// ****************************************************************************

