// ****************************************************************************
// GlottalImageExplorer.
// Copyright (C) 2015-2016 Peter Birkholz.
// This program is free and open-source software.
// ****************************************************************************

#include <wx/file.h>
#include <wx/dcmemory.h>
#include <wx/dir.h>
#include <wx/busyinfo.h>
#include <wx/rawbmp.h>
#include "Film.h"
#include <vfw.h>
#include <cstdlib>


// ****************************************************************************
/// Constructor.
// ****************************************************************************

Film::Film()
{
  images = NULL;
  rawData = NULL;
  isRawData = true;
  numFrames = 0;
}

// ****************************************************************************
/// Loads a film.
// ****************************************************************************

bool Film::loadRawFilm(wxString fileName)
{
  wxFile file(fileName.c_str());
  if (file.IsOpened() == false)
  {
    printf("Failed to open file %s.\n", fileName.c_str());
    return false;
  }


  clear();
  isRawData = true;

  wxBusyInfo wait("Please wait while loading the images...");

  size_t length = file.Length();
  rawData = new (std::nothrow) unsigned char[length];

  if (rawData == NULL)
  {
    wxMessageBox("Failed to allocate memory for raw data.", "Error");
    return false;
  }

  file.Read((void*)rawData, length);
  numFrames = (int)(length / (128 * 256));
  
  if (numFrames > MAX_FRAMES)
  {
    printf("%d frames in the file, but only the maximal number of %d frames is loaded.\n", 
      numFrames, MAX_FRAMES);
    numFrames = MAX_FRAMES;
  }
  else
  {
    printf("%d frames in the file.\n", numFrames);
  }

  // File is closed automatically in the destructor of wxFile.
  return true;
}


// ****************************************************************************
/// Load the frames from an AVI file.
// ****************************************************************************


bool Film::loadAviFilm(wxString fileName)
{
  AVIFileInit();

  // ****************************************************************
  // Open the AVI file.
  // ****************************************************************

  PAVIFILE avi = NULL;

  if (AVIFileOpen(&avi, fileName.c_str(), OF_READ, NULL) != AVIERR_OK)
  {
    printf("Error: AVI file could not be opened.\n");
    if (avi != NULL)
    {
      AVIFileRelease(avi);
    }
    return false;
  }

  AVIFILEINFO aviInfo;
  AVIFileInfo(avi, &aviInfo, sizeof(AVIFILEINFO));

  printf( 
    "\n=== AVI file information ===\n"
    "Dimension: %d x %d pixel\n"
    "Length: %d frames\n"
    "Max bytes per second: %d\n"
    "Samples per second: %d\n"
    "Streams: %d\n"
    "File Type: %d\n", 
    aviInfo.dwWidth,
    aviInfo.dwHeight,
    aviInfo.dwLength,
    aviInfo.dwMaxBytesPerSec,
    (DWORD)(aviInfo.dwRate / aviInfo.dwScale),
    aviInfo.dwStreams,
    aviInfo.szFileType);

  // ****************************************************************
  // Check if it is a valid AVI file for us.
  // ****************************************************************

  if ((aviInfo.dwWidth != WIDTH) || (aviInfo.dwHeight != HEIGHT))
  {
    printf("Error: AVI file has invalid resolution. We need %d x %d pixels.\n",
      WIDTH, HEIGHT);
    return false;
  }
  
  // ****************************************************************
  // Obtain the address of the avi stream.
  // ****************************************************************

  PAVISTREAM pStream = NULL;

  if (AVIFileGetStream(avi, &pStream, streamtypeVIDEO, 0) != AVIERR_OK)
  {
    printf("Error: Address of avi stream interface could not be obtained!\n");
    if (pStream != NULL)
    {
      AVIStreamRelease(pStream);
    }

    AVIFileExit();
    return false;
  }

  // ****************************************************************
  // Get the starting frame number for the stream and the
  // stream length (in frames).
  // ****************************************************************

  int firstFrameIndex;
  int numAviFrames;

  firstFrameIndex = AVIStreamStart(pStream);
  if (firstFrameIndex == -1)
  {
    printf("Error getting the first frame index inside the AVI stream.\n");
    if (pStream != NULL)
    {
      AVIStreamRelease(pStream);
    }

    AVIFileExit();
    return false;
  }

  // ****************************************************************

  numAviFrames = AVIStreamLength(pStream);
  if (numAviFrames == -1)
  {
    printf("Error getting the number of frames inside the AVI stream.\n");
    if (pStream != NULL)
    {
      AVIStreamRelease(pStream);
    }

    AVIFileExit();
    return false;
  }

  printf("The AVI stream starts at frame %d and has %d frames.\n", 
    firstFrameIndex, numAviFrames);

  // ****************************************************************
  // Get bitmaps from the frames.
  // ****************************************************************

  isRawData = false;
  clear();

  numFrames = numAviFrames - firstFrameIndex;
  images = new wxImage[numFrames];

  wxBusyInfo wait("Please wait while loading the images...");

  
  int i, x;
  int dummy;
  PGETFRAME pFrame = AVIStreamGetFrameOpen(pStream, NULL);
  if (pFrame == NULL)
  {
    printf("Error getting the AVI GetFrame object.\n");
    if (pStream != NULL)
    {
      AVIStreamRelease(pStream);
    }
    AVIFileExit();
    return false;
  }

  // ****************************************************************
  // Run through all frames.
  // ****************************************************************

  for (i = 0; i < numFrames; i++)
  {
    BYTE* pDib = (BYTE*)AVIStreamGetFrame(pFrame, i);

    void *rgbData = malloc(WIDTH * HEIGHT * 3);
    memcpy(rgbData, pDib + sizeof(BITMAPINFOHEADER), WIDTH * HEIGHT * 3);

    // Exchange the red and blue channels, because the color sequence
    // in BMP data is blue-green-red.

    for (x = 0; x < WIDTH*HEIGHT; x++)
    {
      dummy = ((unsigned char*)rgbData)[3 * x];
      ((unsigned char*)rgbData)[3 * x] = ((unsigned char*)rgbData)[3 * x + 2];
      ((unsigned char*)rgbData)[3 * x + 2] = dummy;
    }

    images[i].Create(WIDTH, HEIGHT, (unsigned char*)rgbData);
    images[i] = images[i].Mirror(false);
  }

  // Close the stream

  AVIStreamGetFrameClose(pFrame);
  AVIStreamRelease(pStream);
  AVIFileExit();

  return true;
}


// ****************************************************************************
/// Load a sequence of bmp images.
// ****************************************************************************

bool Film::loadBmpSequence(wxString folderName)
{
  wxArrayString fileNames;
  int numFiles = wxDir::GetAllFiles(folderName, &fileNames, "*.bmp");

  if (numFiles < 1)
  {
    wxMessageBox("No bmp files in the selected folder!", "Error");
    return false;
  }

  if (numFiles > MAX_FRAMES)
  {
    printf("Warning: Only the first %d files of the %d files are loaded!\n");
    numFiles = MAX_FRAMES;
  }

  // ****************************************************************
  // Load the individual files.
  // ****************************************************************

  isRawData = false;
  clear();

  numFrames = numFiles;
  images = new wxImage[numFiles];

  int i;
  bool failed = false;

  wxBusyInfo wait("Please wait while loading the images...");

  for (i = 0; (i < numFiles) && (failed == false); i++)
  {
    if (images[i].LoadFile(fileNames[i]) == false)
    {
      wxMessageBox("Failed to load file " + fileNames[i], "Error");
      failed = true;
    }

    if (failed == false)
    {
      // Check the resolution of the image.
      if ((images[i].GetWidth() != WIDTH) || (images[i].GetHeight() != HEIGHT))
      {
        wxMessageBox("Wrong resolution for the file " + fileNames[i] + 
          ". The resolution must be 256x256.", "Error");
        failed = true;
      }
    }
   
  }

  if (failed == false)
  {
    printf("%d files successfully loaded.\n", numFrames);
  }
  else
  {
    clear();
  }

  return (failed == false);
}


// ****************************************************************************
// ****************************************************************************

wxImage Film::getFrame(int frameIndex)
{
  if ((frameIndex < 0) || (frameIndex >= numFrames))
  {
    return wxImage(WIDTH, HEIGHT);
  }
  else
  if (isRawData)
  {
    return extractFrame(frameIndex);
  }
  else
  {
    return images[frameIndex];
  }
}


// ****************************************************************************
/// Returnes the number of film frames.
// ****************************************************************************

int Film::getNumFrames()
{
  return numFrames;
}


// ****************************************************************************
/// Clear all data of the film.
// ****************************************************************************

void Film::clear()
{
  numFrames = 0;

  if (images != NULL)
  {
    delete[] images;
    images = NULL;
  }

  if (rawData != NULL)
  {
    delete rawData;
    rawData = NULL;
  }

}

// ****************************************************************************
/// Retrieve the frame with the given frame index as image.
// ****************************************************************************

wxImage Film::extractFrame(int frameIndex)
{
  wxImage frame(WIDTH, HEIGHT);

  if ((frameIndex < 0) || (frameIndex >= numFrames))
  {
    return frame;
  }

  int x, y;
  int rawOffset = frameIndex*128*256;
  unsigned char *imageData = frame.GetData();
  unsigned char rawByte;
  unsigned char *pixelPtr;
  int imageX, imageY;

  // ****************************************************************
  // Distribute the single color bytes in the raw data into the
  // target image RGB-array.
  // ****************************************************************

  for (x=0; x < 256; x++)
  {
    for (y=0; y < 128; y++)
    {
      rawByte = rawData[rawOffset + y*256 + x];
      imageX = x;
      imageY = 2*y + ((x+1) & 1);
      pixelPtr = imageData + ((imageY*256) + imageX)*3;

      // Which color component is set?
      if ((x & 1) == 0)
      {
        // Blue
        if (((x/2) & 1) == (y & 1))
        {
          *(pixelPtr + 2) = rawByte;
        }
        else
        // Red
        {
          *pixelPtr = rawByte;
        }
      }
      else
      {
        // Green
        *(pixelPtr + 1) = rawByte;
      }
    }
  }

  // ****************************************************************
  // Add the RED components in the center of 4 original RED pixels.
  // ****************************************************************

  unsigned char *p;

  for (y=3; y <= 251; y+=4)
  {
    p = imageData + y*256*3 + 2*3;
    for (x=2; x <= 250; x+=4)
    {
      *p = (unsigned char)(((int)*(p-2*3) + (int)*(p+2*3) + (int)*(p-256*2*3) + (int)*(p+256*2*3)) / 4);
      p+= 4*3;
    }
  }

  for (y=5; y <= 253; y+=4)
  {
    p = imageData + y*256*3 + 4*3;
    for (x=4; x <= 252; x+=4)
    {
      *p = (unsigned char)(((int)*(p-2*3) + (int)*(p+2*3) + (int)*(p-256*2*3) + (int)*(p+256*2*3)) / 4);
      p+= 4*3;
    }
  }

  // Left margin
  p = imageData + 5*256*3;
  for (y=5; y <= 253; y+=4)
  {
    *p = (unsigned char)(((int)*(p+2*3) + (int)*(p-256*2*3) + (int)*(p+256*2*3)) / 3);
    p+= 4*256*3;
  }
  
  // Right margin
  p = imageData + 3*256*3 + 254*3;
  for (y=3; y <= 251; y+=4)
  {
    *p = (unsigned char)(((int)*(p-2*3) + (int)*(p-256*2*3) + (int)*(p+256*2*3)) / 3);
    p+= 4*256*3;
  }

  // Top margin
  p = imageData + 1*256*3 + 4*3;
  for (x=4; x <= 252; x+=4)
  {
    *p = (unsigned char)(((int)*(p-2*3) + (int)*(p+2*3) + (int)*(p+256*2*3)) / 3);
    p+= 4*3;
  }

  // Bottom margin
  p = imageData + 255*256*3 + 2*3;
  for (x=2; x <= 250; x+=4)
  {
    *p = (unsigned char)(((int)*(p-2*3) + (int)*(p+2*3) + (int)*(p-256*2*3)) / 3);
    p+= 4*3;
  }

  // Top-left pixel
  p = imageData + 1*256*3 + 0*3;
  *p = (unsigned char)(((int)*(p+2*3) + (int)*(p+256*2*3)) / 2);

  // Bottom-right pixel
  p = imageData + 255*256*3 + 254*3;
  *p = (unsigned char)(((int)*(p-2*3) + (int)*(p-256*2*3)) / 2);

  // ****************************************************************
  // Add the BLUE components in the center of 4 original BLUE pixels.
  // ****************************************************************

  for (y=5; y <= 253; y+=4)
  {
    p = imageData + y*256*3 + 2*3 + 2;    // +2, becaus it's BLUE
    for (x=2; x <= 250; x+=4)
    {
      *p = (unsigned char)(((int)*(p-2*3) + (int)*(p+2*3) + (int)*(p-256*2*3) + (int)*(p+256*2*3)) / 4);
      p+= 4*3;
    }
  }

  for (y=3; y <= 251; y+=4)
  {
    p = imageData + y*256*3 + 4*3 + 2;    // +2, becaus it's BLUE
    for (x=4; x <= 252; x+=4)
    {
      *p = (unsigned char)(((int)*(p-2*3) + (int)*(p+2*3) + (int)*(p-256*2*3) + (int)*(p+256*2*3)) / 4);
      p+= 4*3;
    }
  }

  // Left margin
  p = imageData + 3*256*3 + 2;    // +2, becaus it's BLUE
  for (y=3; y <= 251; y+=4)
  {
    *p = (unsigned char)(((int)*(p+2*3) + (int)*(p-256*2*3) + (int)*(p+256*2*3)) / 3);
    p+= 4*256*3;
  }
  
  // Right margin
  p = imageData + 5*256*3 + 254*3 + 2;    // +2, becaus it's BLUE
  for (y=5; y <= 253; y+=4)
  {
    *p = (unsigned char)(((int)*(p-2*3) + (int)*(p-256*2*3) + (int)*(p+256*2*3)) / 3);
    p+= 4*256*3;
  }

  // Top margin
  p = imageData + 1*256*3 + 2*3 + 2;    // +2, becaus it's BLUE
  for (x=2; x <= 250; x+=4)
  {
    *p = (unsigned char)(((int)*(p-2*3) + (int)*(p+2*3) + (int)*(p+256*2*3)) / 3);
    p+= 4*3;
  }

  // Bottom margin
  p = imageData + 255*256*3 + 4*3 + 2;    // +2, becaus it's BLUE
  for (x=4; x <= 252; x+=4)
  {
    *p = (unsigned char)(((int)*(p-2*3) + (int)*(p+2*3) + (int)*(p-256*2*3)) / 3);
    p+= 4*3;
  }

  // Top-right pixel
  p = imageData + 1*256*3 + 254*3 + 2;    // +2, becaus it's BLUE
  *p = (unsigned char)(((int)*(p-2*3) + (int)*(p+256*2*3)) / 2);

  // Bottom-left pixel
  p = imageData + 255*256*3 + 0*3 + 2;    // +2, becaus it's BLUE
  *p = (unsigned char)(((int)*(p+2*3) + (int)*(p-256*2*3)) / 2);


  // ****************************************************************
  // Interpolate missing red and blue pixels in every 2nd horizontal 
  // line.
  // ****************************************************************

  for (y=1; y <= 255; y+=2)
  {
    p = imageData + y*256*3 + 1*3;
    for (x=1; x <= 253; x+=2)
    {
      *p = (unsigned char)(((int)*(p-3) + (int)*(p+3)) / 2);    // RED
      *(p+2) = (unsigned char)(((int)*(p-3+2) + (int)*(p+3+2)) / 2);    // BLUE
      p+= 2*3;
    }
    // Last pixel in the row
    p = imageData + y*256*3 + 255*3;
    *p = *(p-3);
    *(p+2) = *(p-3+2);
  }

  // ****************************************************************
  // Interpolate red and blue in the empty horizontal lines.
  // ****************************************************************

  // Uppermost line
  p = imageData;
  for (x=0; x <= 255; x++)
  {
    *p = *(p + 256*3);          // RED
    *(p+2) = *(p + 256*3 + 2);  // BLUE
    p+= 3;    
  }

  for (y=2; y <= 254; y+=2)
  {
    p = imageData + y*256*3;
    for (x=0; x <= 255; x++)
    {
      *p = (unsigned char)(((int)*(p-256*3) + (int)*(p+256*3)) / 2);        // RED
      *(p+2) = (unsigned char)(((int)*(p-256*3+2) + (int)*(p+256*3+2)) / 2);  // BLUE
      p+= 3;
    }
  }

  // ****************************************************************
  // Interpolate missing GREEN pixels in every 2nd horizontal line.
  // ****************************************************************

  for (y=0; y <= 254; y+=2)
  {
    // First pixel in the row
    p = imageData + y*256*3 + 0*3 + 1;      // +1, because it's GREEN
    *p = *(p+3);

    p = imageData + y*256*3 + 2*3 + 1;      // +1, because it's GREEN
    for (x=2; x <= 254; x+=2)
    {
      *p = (unsigned char)(((int)*(p-3) + (int)*(p+3)) / 2);
      p+= 2*3;
    }
  }

  // ****************************************************************
  // Interpolate GREEN in the empty horizontal lines.
  // ****************************************************************

  for (y=1; y <= 253; y+=2)
  {
    p = imageData + y*256*3 + 1;      // +1, because it's GREEN
    for (x=0; x <= 255; x++)
    {
      *p = (unsigned char)(((int)*(p-256*3) + (int)*(p+256*3)) / 2);
      p+= 3;
    }
  }

  // Lowermost line
  p = imageData + 255*256 + 1;      // +1, because it's GREEN
  for (x=0; x <= 255; x++)
  {
    *p = *(p - 256*3);
    p+= 3;    
  }

  return frame;
}


// ****************************************************************************
// ****************************************************************************
